<?php

use Illuminate\Foundation\Testing\WithoutMiddleware;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use ToStartWork\Mail\UserAppliedToCourse;

class CourseTest extends TestCase
{
    use DatabaseMigrations;

    public function setUp()
    {
        parent::setUp();
        $this->withoutMiddleware();
    }
    
    public function testCourseList()
    {
        factory(\ToStartWork\Course::class, 5)->create();

        $response = $this->json('GET', '/api/course/list');

        $response->assertResponseOk();
        $courses = json_decode($this->response->content());

        $this->assertEquals(5, count(json_decode($this->response->content())));
        $this->assertObjectHasAttribute('category', $courses[0]);
        $this->assertObjectHasAttribute('provider', $courses[0]);
    }

    public function testEventsFiredWhenUserAppliesToCourse()
    {
        $this->expectsEvents(\ToStartWork\Events\UserAppliedToCourse::class);
        $user = factory(\ToStartWork\User::class, 1)->create();
        $course = factory(\ToStartWork\Course::class, 1)->create();

        $response = $this->actingAs($user)->json('POST', '/api/course/apply', ['course_id' => $course->id]);
    }

    public function testUserApplyToCourse()
    {
        Mail::fake();

        $user = factory(\ToStartWork\User::class, 1)->create();
        $course = factory(\ToStartWork\Course::class, 1)->create();

        $response = $this->actingAs($user)->json('POST', '/api/course/apply', ['course_id' => $course->id]);

        $this->seeInDatabase('course_user', [
            'user_id' => $user->id,
            'course_id' => $course->id,
        ]);

        Mail::assertSentTo([$course->courseProvider->contact_email], \ToStartWork\Mail\UserAppliedToCourse::class);

        Mail::assertSent(UserAppliedToCourse::class, function ($mail) use ($user, $course) {
            return $mail->course->id === $course->id && $mail->user->id == $user->id;
        });
    }

    public function testUserApplyToCourseRealEmail()
    {
        $user = factory(\ToStartWork\User::class, 1)->create();
        $course = factory(\ToStartWork\Course::class, 1)->create();

        $response = $this->actingAs($user)->json('POST', '/api/course/apply', ['course_id' => $course->id]);

        $this->seeInDatabase('course_user', [
            'user_id' => $user->id,
            'course_id' => $course->id,
        ]);

        $this->assertResponseOk();
    }
}
