<?php

namespace ToStartWork\Http\Controllers;

use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;
use League\Csv\Reader;
use ToStartWork\CourseCategory;
use ToStartWork\CourseProvider;
use ToStartWork\Http\Requests\Admin\CourseManagement\StoreCourseRequest;
use ToStartWork\Http\Requests\Admin\CourseManagement\UpdateCourseRequest;
use ToStartWork\Http\Requests\Admin\UserManagement\StoreUserRequest;
use ToStartWork\Http\Requests\Admin\UserManagement\UpdateUserRequest;
use ToStartWork\Notification;
use ToStartWork\User;
use ToStartWork\Course;

class CourseController extends Controller
{
    protected $providerOptions;
    protected $categoryOptions;

    function __construct()
    {
        $this->providerOptions = CourseProvider::orderBy('company', 'desc')->get()->pluck('company', 'id');
        $this->categoryOptions = CourseCategory::orderBy('name', 'desc')->get()->pluck('name', 'id');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $courses = Course::orderBy('name', 'desc')->get();

        return view('courses/index', compact('courses'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('courses/create', [
            'providerOptions' => $this->providerOptions,
            'categoryOptions' => $this->categoryOptions
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreCourseRequest $request)
    {
        Course::create($request->except('_token'));

        return Redirect::route('courses.index')->with('status', 'Created Course');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Course $course
     * @return \Illuminate\Http\Response
     * @internal param Notification $notification
     * @internal param int $id
     */
    public function edit(Course $course)
    {
        return view('courses.edit', [
            'providerOptions' => $this->providerOptions,
            'categoryOptions' => $this->categoryOptions,
            'course' => $course
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request|UpdateCourseRequest $request
     * @param Course $course
     * @return \Illuminate\Http\Response
     * @internal param int $id
     */
    public function update(UpdateCourseRequest $request, Course $course)
    {
        $course->update($request->except('_token'));

        return Redirect::route('courses.index')->with('status', 'Updated Course');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Course $course)
    {
        $course->delete();

        return response('Okay', 200);
    }

    public function getTemplate()
    {
        return response()->download(storage_path('files/course_template.csv'));
    }

    public function getCategoryAndProviderIds()
    {
        $csv = fopen(storage_path('ids.csv'), 'w');

        fputcsv($csv, ['Provider ID', 'Provider Name']);
        foreach ($this->providerOptions as $id => $name) {
            fputcsv($csv, [$id, $name]);
        }

        fputcsv($csv, ['', '']);  // insert separation line

        fputcsv($csv, ['Category ID', 'Category Name']);
        foreach ($this->categoryOptions as $id => $name) {
            fputcsv($csv, [$id, $name]);
        }

        fclose($csv);

        return response()->download(storage_path('ids.csv'));
    }

    public function uploadCsv(Request $request)
    {
        try {
            return DB::transaction(function () use ($request) {
                $path = $request->file('csv')->storeAs('csv', 'course.csv');

                $csvReader = Reader::createFromPath(storage_path("app/$path"));
                $rows = $csvReader->fetch();

                $data = [];

                foreach ($rows as $key => $row) {
                    // remove csv header
                    if ($key === 0) {
                        continue;
                    }

                    $data[] = [
                        'name' => array_get($row, 0),
                        'provider_id' => array_get($row, 1),
                        'category_id' => array_get($row, 2),
                        'location' => array_get($row, 3),
                        'enroll_date' => array_get($row, 4),
                        'price' => array_get($row, 5),
                        'citb_price' => array_get($row, 6),
                        'active' => trim(strtolower(array_get($row, 7, ''))) === 'yes' ? 1 : 0,
                        'description' => array_get($row, 8)
                    ];

                    // chunk insert data to DB
                    if (count($data) === 500) {
                        DB::table('courses')->insert($data);
                        $data = [];
                    }
                }

                // insert the rest of the data
                if (count($data)) {
                    DB::table('courses')->insert($data);
                }

                return response()->json();
            });
        } catch (\Exception $e) {
            $date = date('Y-m-d H:i:s');
            \Log::error("[$date]: {$e->getMessage()}");
            return response()->json($e->getMessage(), 500);
        }
    }
}
