<?php

defined( 'ABSPATH' ) or die;

$GLOBALS['processed_terms'] = array();
$GLOBALS['processed_posts'] = array();

require_once ABSPATH . 'wp-admin/includes/post.php';
require_once ABSPATH . 'wp-admin/includes/taxonomy.php';
require_once ABSPATH . 'wp-admin/includes/image.php';

/**
 * Add an Import Action, this data is stored for processing after import is done.
 *
 * Each action is sent as an Ajax request and is handled by themify-ajax.php file
 */ 
function themify_add_import_action( $action = '', $data = array() ) {
	global $import_actions;

	if ( ! isset( $import_actions[ $action ] ) ) {
		$import_actions[ $action ] = array();
	}

	$import_actions[ $action ][] = $data;
}

function themify_import_post( $post ) {
	global $processed_posts, $processed_terms;

	if ( ! post_type_exists( $post['post_type'] ) ) {
		return;
	}

	/* Menu items don't have reliable post_title, skip the post_exists check */
	if( $post['post_type'] !== 'nav_menu_item' ) {
		$post_exists = post_exists( $post['post_title'], '', $post['post_date'] );
		if ( $post_exists && get_post_type( $post_exists ) == $post['post_type'] ) {
			$processed_posts[ intval( $post['ID'] ) ] = intval( $post_exists );
			return;
		}
	}

	if( $post['post_type'] == 'nav_menu_item' ) {
		if( ! isset( $post['tax_input']['nav_menu'] ) || ! term_exists( $post['tax_input']['nav_menu'], 'nav_menu' ) ) {
			return;
		}
		$_menu_item_type = $post['meta_input']['_menu_item_type'];
		$_menu_item_object_id = $post['meta_input']['_menu_item_object_id'];

		if ( 'taxonomy' == $_menu_item_type && isset( $processed_terms[ intval( $_menu_item_object_id ) ] ) ) {
			$post['meta_input']['_menu_item_object_id'] = $processed_terms[ intval( $_menu_item_object_id ) ];
		} else if ( 'post_type' == $_menu_item_type && isset( $processed_posts[ intval( $_menu_item_object_id ) ] ) ) {
			$post['meta_input']['_menu_item_object_id'] = $processed_posts[ intval( $_menu_item_object_id ) ];
		} else if ( 'custom' != $_menu_item_type ) {
			// associated object is missing or not imported yet, we'll retry later
			// $missing_menu_items[] = $item;
			return;
		}
	}

	$post_parent = ( $post['post_type'] == 'nav_menu_item' ) ? $post['meta_input']['_menu_item_menu_item_parent'] : (int) $post['post_parent'];
	$post['post_parent'] = 0;
	if ( $post_parent ) {
		// if we already know the parent, map it to the new local ID
		if ( isset( $processed_posts[ $post_parent ] ) ) {
			if( $post['post_type'] == 'nav_menu_item' ) {
				$post['meta_input']['_menu_item_menu_item_parent'] = $processed_posts[ $post_parent ];
			} else {
				$post['post_parent'] = $processed_posts[ $post_parent ];
			}
		}
	}

	/**
	 * for hierarchical taxonomies, IDs must be used so wp_set_post_terms can function properly
	 * convert term slugs to IDs for hierarchical taxonomies
	 */
	if( ! empty( $post['tax_input'] ) ) {
		foreach( $post['tax_input'] as $tax => $terms ) {
			if( is_taxonomy_hierarchical( $tax ) ) {
				$terms = explode( ', ', $terms );
				$post['tax_input'][ $tax ] = array_map( 'themify_get_term_id_by_slug', $terms, array_fill( 0, count( $terms ), $tax ) );
			}
		}
	}

	$post['post_author'] = (int) get_current_user_id();
	$post['post_status'] = 'publish';

	$old_id = $post['ID'];

	unset( $post['ID'] );
	$post_id = wp_insert_post( $post, true );
	if( is_wp_error( $post_id ) ) {
		return false;
	} else {
		$processed_posts[ $old_id ] = $post_id;

		return $post_id;
	}
}

function themify_get_placeholder_image() {
	static $placeholder_image = null;

	if ( $placeholder_image == null ) {
		if ( ! function_exists( 'WP_Filesystem' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}
		WP_Filesystem();
		global $wp_filesystem;

		$post = array();

		$upload = wp_upload_bits( 'themify-placeholder.jpg', null, $wp_filesystem->get_contents( THEMIFY_DIR . '/img/image-placeholder.jpg' ) );

		if ( $info = wp_check_filetype( $upload['file'] ) )
			$post = array(
				'post_mime_type' => $info['type']
			);
		else
			return new WP_Error( 'attachment_processing_error', __( 'Invalid file type', 'themify' ) );

		$post['guid'] = $upload['url'];
		$placeholder_image = wp_insert_attachment( $post, $upload['file'] );
		require_once( ABSPATH . 'wp-admin/includes/image.php' );
		wp_update_attachment_metadata( $placeholder_image, wp_generate_attachment_metadata( $placeholder_image, $upload['file'] ) );
	}

	return $placeholder_image;
}

function themify_import_term( $term ) {
	global $processed_terms;

	if( $term_id = term_exists( $term['slug'], $term['taxonomy'] ) ) {
		if ( is_array( $term_id ) ) $term_id = $term_id['term_id'];
		if ( isset( $term['term_id'] ) )
			$processed_terms[ intval( $term['term_id'] ) ] = (int) $term_id;
		return (int) $term_id;
	}

	if ( empty( $term['parent'] ) || ! isset( $processed_terms[ intval( $term['parent'] ) ] ) ) {
		$parent = 0;
	} else {
		$parent = term_exists( $processed_terms[ intval( $term['parent'] ) ], $term['taxonomy'] );
		if ( is_array( $parent ) ) $parent = $parent['term_id'];
	}

	$id = wp_insert_term( $term['name'], $term['taxonomy'], array(
		'parent' => $parent,
		'slug' => $term['slug'],
		'description' => $term['description'],
	) );
	if ( ! is_wp_error( $id ) ) {
		if ( isset( $term['term_id'] ) ) {
			// success!
			$processed_terms[ intval($term['term_id']) ] = $id['term_id'];
			if ( isset( $term['thumbnail'] ) ) {
				themify_add_import_action( 'term_thumb', array(
					'id' => $id['term_id'],
					'thumb' => $term['thumbnail'],
				) );
			}
			return $term['term_id'];
		}
	}

	return false;
}

function themify_get_term_id_by_slug( $slug, $tax ) {
	$term = get_term_by( 'slug', $slug, $tax );
	if( $term ) {
		return $term->term_id;
	}

	return false;
}

function themify_undo_import_term( $term ) {
	$term_id = term_exists( $term['slug'], $term['taxonomy'] );
	if ( $term_id ) {
		if ( is_array( $term_id ) ) $term_id = $term_id['term_id'];

		if ( $term_thumbnail = get_term_meta( $term['term_id'], 'thumbnail_id', true ) ) {
			wp_delete_attachment( $term_thumbnail, true );
		}

		if ( isset( $term_id ) ) {
			wp_delete_term( $term_id, $term['taxonomy'] );
		}
	}
}

/**
 * Determine if a post exists based on title, content, and date
 *
 * @global wpdb $wpdb WordPress database abstraction object.
 *
 * @param array $args array of database parameters to check
 * @return int Post ID if post exists, 0 otherwise.
 */
function themify_post_exists( $args = array() ) {
	global $wpdb;

	$query = "SELECT ID FROM $wpdb->posts WHERE 1=1";
	$db_args = array();

	foreach ( $args as $key => $value ) {
		$value = wp_unslash( sanitize_post_field( $key, $value, 0, 'db' ) );
		if( ! empty( $value ) ) {
			$query .= ' AND ' . $key . ' = %s';
			$db_args[] = $value;
		}
	}

	if ( !empty ( $args ) )
		return (int) $wpdb->get_var( $wpdb->prepare($query, $args) );

	return 0;
}

function themify_undo_import_post( $post ) {
	if( $post['post_type'] == 'nav_menu_item' ) {
		$post_exists = themify_post_exists( array(
			'post_name' => $post['post_name'],
			'post_modified' => $post['post_date'],
			'post_type' => 'nav_menu_item',
		) );
	} else {
		$post_exists = post_exists( $post['post_title'], '', $post['post_date'] );
	}
	if( $post_exists && get_post_type( $post_exists ) == $post['post_type'] ) {
		/**
		 * check if the post has been modified, if so leave it be
		 *
		 * NOTE: posts are imported using wp_insert_post() which modifies post_modified field
		 * to be the same as post_date, hence to check if the post has been modified,
		 * the post_modified field is compared against post_date in the original post.
		 */
		if( $post['post_date'] == get_post_field( 'post_modified', $post_exists ) ) {
			// find and remove all post attachments
			$attachments = get_posts( array(
				'post_type' => 'attachment',
				'posts_per_page' => -1,
				'post_parent' => $post_exists,
			) );
			if ( $attachments ) {
				foreach ( $attachments as $attachment ) {
					wp_delete_attachment( $attachment->ID, true );
				}
			}
			wp_delete_post( $post_exists, true ); // true: bypass trash
		}
	}
}

function themify_process_post_import( $post ) {
	if( ERASEDEMO ) {
		themify_undo_import_post( $post );
	} else {
		if ( $id = themify_import_post( $post ) ) {
			if ( defined( 'IMPORT_IMAGES' ) && ! IMPORT_IMAGES ) {
				/* if importing images is disabled and post is supposed to have a thumbnail, create a placeholder image instead */
				if ( isset( $post['thumb'] ) ) { // the post is supposed to have featured image
					$placeholder = themify_get_placeholder_image();
					if ( ! empty( $placeholder ) && ! is_wp_error( $placeholder ) ) {
						set_post_thumbnail( $id, $placeholder );
					}
				}
			} else {
				if ( isset( $post["thumb"] ) ) {
					themify_add_import_action( 'post_thumb', array(
						'id' => $id,
						'thumb' => $post["thumb"],
					) );
				}
				if ( isset( $post["gallery_shortcode"] ) ) {
					themify_add_import_action( 'gallery_field', array(
						'id' => $id,
						'fields' => $post["gallery_shortcode"],
					) );
				}
				if ( isset( $post["_product_image_gallery"] ) ) {
					themify_add_import_action( 'product_gallery', array(
						'id' => $id,
						'images' => $post["_product_image_gallery"],
					) );
				}
			}
		}
	}
}
$thumbs = array();
function themify_do_demo_import() {
global $import_actions;

	if ( isset( $GLOBALS["ThemifyBuilder_Data_Manager"] ) ) {
		remove_action( "save_post", array( $GLOBALS["ThemifyBuilder_Data_Manager"], "save_builder_text_only"), 10, 3 );
	}
$term = array (
  'term_id' => 2,
  'name' => 'Main Navigation',
  'slug' => 'main-navigation',
  'term_group' => 0,
  'taxonomy' => 'nav_menu',
  'description' => '',
  'parent' => 0,
);

if( ERASEDEMO ) {
	themify_undo_import_term( $term );
} else {
	themify_import_term( $term );
}

$term = array (
  'term_id' => 3,
  'name' => 'Footer Widget 1',
  'slug' => 'footer-widget-1',
  'term_group' => 0,
  'taxonomy' => 'nav_menu',
  'description' => '',
  'parent' => 0,
);

if( ERASEDEMO ) {
	themify_undo_import_term( $term );
} else {
	themify_import_term( $term );
}

$term = array (
  'term_id' => 4,
  'name' => 'Footer Widget 2',
  'slug' => 'footer-widget-2',
  'term_group' => 0,
  'taxonomy' => 'nav_menu',
  'description' => '',
  'parent' => 0,
);

if( ERASEDEMO ) {
	themify_undo_import_term( $term );
} else {
	themify_import_term( $term );
}

$post = array (
  'ID' => 173,
  'post_date' => '2019-07-28 10:00:09',
  'post_date_gmt' => '2019-07-28 10:00:09',
  'post_content' => '<!-- wp:themify-builder/canvas /--><!--themify_builder_static--><h1>Start a happier and healthier life today.</h1> <h3>Our trainers will help you get where</h3> <h3>you want to be. </h3>
<p>Whether you have a personal goal, a race event you need to get ready for, or you simply want to live better, one thing is for sure - our trainers won\'t give up on you. </p>
<h3>About Us</h3> <p>Since 2012, we\'ve been providing the downtown Toronto core with our physical trainers, workout classes, and wellness workshops. We\'ve been expanding and have recently added a third location.</p> <p>That\'s why we chose the Ultra Trainers skin to represent our team of personal trainers and showcase our gym. Ultra Trainers is the answer to your search for the best gym, personal trainer, and fitness coaching website. </p>
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/leg-excersise.jpg" width="555 " alt="leg-excersise" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/leg-excersise.jpg 555w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/leg-excersise-300x267.jpg 300w" sizes="(max-width: 555px) 100vw, 555px" />
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/push-up-1.jpg" width="555 " alt="push-up" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/push-up-1.jpg 555w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/push-up-1-300x240.jpg 300w" sizes="(max-width: 555px) 100vw, 555px" />
<h3>Our Vision</h3> <p>The majority of us have suffered some type of physical injury, trauma, or even emotional wounds. We want to help you overcome your fears and power forward to the future, strong, you!</p> <p>Our passion remains in creating positive environments that are welcoming, supportive, and are a force for change and wellness. Safety is always first.</p>
<h3>Why Us?<br/></h3>
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/goal-icon.png" alt="goal-icon" />
Analyze Your Goal
<p>We assess and evaluate how to get you where you want to be - even if we have to start slow, we\\\'ll get there together.</p>
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/work-out-icon.png" alt="work-out-icon" />
Show Your Work
<p>We\'ll guide you every step of the way, but show us your work. Let\'s see your dedication and commitment in action.</p>
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/performance-icon.png" alt="performance-icon" />
Improve Your Performance
<p>Let\'s take a step back, re-assess, analyze, and re-commit to a the same or revised goal.</p>
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/achievement-icon.png" alt="achievement-icon" />
Achieve Your Destiny
<p>After all the hard work, sweat, and blood, feel the amazing and sweet feeling of victory. You did it!</p>
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/adidas-logo.png" alt="adidas-logo" />
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/nike-logo.png" alt="nike-logo" />
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/reebok.png" alt="reebok" />
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/converse-logo.png" alt="converse-logo" />
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/under-armour.png" alt="under-armour" />
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/puma.png" alt="puma" />
<h3>Our Trainers<br/>professionally trained </h3>
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/bernard-260x260.jpg" width="260" height="260" title="Benji Jr." alt="CrossFit Coach" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/bernard.jpg 260w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/bernard-150x150.jpg 150w" sizes="(max-width: 260px) 100vw, 260px" /> <h3> Benji Jr. </h3> CrossFit Coach
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/monica-rodriguez-260x260.png" width="260" height="260" title="Jessica Kim" alt="Zumba Coach" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/monica-rodriguez-260x260.png 260w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/monica-rodriguez-150x150.png 150w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/monica-rodriguez-250x250.png 250w" sizes="(max-width: 260px) 100vw, 260px" /> <h3> Jessica Kim </h3> Zumba Coach
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jessica-260x260.jpg" width="260" height="260" title="Shania Dee" alt="Booty Boot Camp" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jessica.jpg 260w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/jessica-150x150.jpg 150w" sizes="(max-width: 260px) 100vw, 260px" /> <h3> Shania Dee </h3> Booty Boot Camp
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/robert-moore-260x260.jpg" width="260" height="260" title="Robert Moore" alt="Muay Thai" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/robert-moore.jpg 260w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/robert-moore-150x150.jpg 150w" sizes="(max-width: 260px) 100vw, 260px" /> <h3> Robert Moore </h3> Muay Thai
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/michelle-kim-260x260.jpg" width="260" height="260" title="Michelina Jung" alt="Pilates" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/michelle-kim.jpg 260w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/michelle-kim-150x150.jpg 150w" sizes="(max-width: 260px) 100vw, 260px" /> <h3> Michelina Jung </h3> Pilates
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/michael-260x260.jpg" width="260" height="260" title="Michael Okue" alt="Spinning" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/michael.jpg 260w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/michael-150x150.jpg 150w" sizes="(max-width: 260px) 100vw, 260px" /> <h3> Michael Okue </h3> Spinning<!--/themify_builder_static-->',
  'post_title' => 'About',
  'post_excerpt' => '',
  'post_name' => 'about',
  'post_modified' => '2020-02-19 07:30:39',
  'post_modified_gmt' => '2020-02-19 07:30:39',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?page_id=173',
  'menu_order' => 0,
  'post_type' => 'page',
  'meta_input' => 
  array (
    'page_layout' => 'sidebar-none',
    'content_width' => 'full_width',
    'hide_page_title' => 'yes',
    'section_scrolling_mobile' => 'on',
    'mobile_menu_styles' => 'default',
    'header_wrap' => 'transparent',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"nyd8923\\",\\"cols\\":[{\\"element_id\\":\\"9wul157\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"cekj157\\",\\"mod_settings\\":{\\"content_text\\":\\"<h1>Start a happier and healthier life today.<\\\\/h1>\\\\n<h3>Our trainers will help you get where<\\\\/h3>\\\\n<h3>you want to be. <\\\\/h3>\\",\\"margin_bottom\\":\\"40\\",\\"breakpoint_mobile\\":{\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"18\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\"}}}],\\"styling\\":{\\"font_color\\":\\"#ffffff\\",\\"breakpoint_tablet\\":{\\"padding_left\\":\\"43\\",\\"padding_left_unit\\":\\"px\\"},\\"breakpoint_tablet_landscape\\":{\\"padding_left\\":\\"17\\",\\"padding_left_unit\\":\\"px\\"}}},{\\"element_id\\":\\"0mkt926\\",\\"grid_class\\":\\"col4-2\\"}],\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-3-4-1\\",\\"styling\\":{\\"row_width\\":\\"fullwidth\\",\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bg-about.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"cover_color\\":\\"#0e2259_0.53\\",\\"cover_color-type\\":\\"color\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom\\":\\"18\\",\\"padding_top\\":\\"18\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"bottom-frame_height\\":\\"30\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ecf3fd\\",\\"bottom-frame_layout\\":\\"slant1\\",\\"padding_opp_top\\":\\"1\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"32\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top\\":\\"32\\",\\"padding_top_unit\\":\\"%\\"}}},{\\"element_id\\":\\"aitv715\\",\\"cols\\":[{\\"element_id\\":\\"topg716\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"x0oy407\\",\\"mod_settings\\":{\\"content_text\\":\\"<p>Whether you have a personal goal, a race event you need to get ready for, or you simply want to live better, one thing is for sure - our trainers won\\\'t give up on you. <\\\\/p>\\",\\"text_align\\":\\"center\\",\\"line_height_unit\\":\\"em\\",\\"line_height\\":\\"1.6\\",\\"font_size_unit\\":\\"px\\",\\"font_size\\":\\"25\\",\\"font_color_type\\":\\"font_color_solid\\",\\"font_color\\":\\"#000000\\",\\"breakpoint_mobile\\":{\\"text_align\\":\\"center\\",\\"line_height_unit\\":\\"em\\",\\"line_height\\":\\"1.6\\",\\"font_size_unit\\":\\"px\\",\\"font_size\\":\\"20\\",\\"font_color\\":\\"#000000\\",\\"font_color_type\\":\\"font_color_solid\\"}}}]}],\\"styling\\":{\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"3\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"6\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"background_color\\":\\"ecf3fd\\",\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\"}},{\\"element_id\\":\\"9mvr617\\",\\"cols\\":[{\\"element_id\\":\\"ipar619\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"element_id\\":\\"xyzg697\\",\\"cols\\":[{\\"element_id\\":\\"0ngq698\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"f970186\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>About Us<\\\\/h3>\\\\n<p>Since 2012, we\\\'ve been providing the downtown Toronto core with our physical trainers, workout classes, and wellness workshops. We\\\'ve been expanding and have recently added a third location.<\\\\/p>\\\\n<p>That\\\'s why we chose the Ultra Trainers skin to represent our team of personal trainers and showcase our gym. Ultra Trainers is the answer to your search for the best gym, personal trainer, and fitness coaching website. <\\\\/p>\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"36\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"breakpoint_mobile\\":{\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"30\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"margin_bottom\\":\\"40\\"}}}]},{\\"element_id\\":\\"96v2698\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"9fau220\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/leg-excersise.jpg\\",\\"style_image\\":\\"image-top\\",\\"width_image\\":\\"555 \\",\\"custom_parallax_scroll_speed\\":\\"2\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"styling\\":{\\"margin_bottom\\":\\"10\\",\\"margin_bottom_unit\\":\\"%\\"}},{\\"element_id\\":\\"9kjf333\\",\\"cols\\":[{\\"element_id\\":\\"5fxy333\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"8g0l334\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/push-up-1.jpg\\",\\"style_image\\":\\"image-top\\",\\"width_image\\":\\"555 \\",\\"custom_parallax_scroll_speed\\":\\"1\\"}}]},{\\"element_id\\":\\"s117334\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"8hpf334\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Our Vision<\\\\/h3>\\\\n<p>The majority of us have suffered some type of physical injury, trauma, or even emotional wounds. We want to help you overcome your fears and power forward to the future, strong, you!<\\\\/p>\\\\n<p>Our passion remains in creating positive environments that are welcoming, supportive, and are a force for change and wellness. Safety is always first.<\\\\/p>\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"36\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"breakpoint_mobile\\":{\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"30\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"margin_top\\":\\"40\\",\\"font_color_type\\":\\"font_color_solid\\"}}}]}],\\"column_alignment\\":\\"col_align_middle\\"}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"styling\\":{\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ecf3fd\\",\\"top-frame_layout\\":\\"slant2\\",\\"top-frame_type\\":\\"top-presets\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"14\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"14\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_color\\":\\"#ecf3fd\\",\\"bottom-frame_layout\\":\\"slant2\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"23\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top\\":\\"23\\",\\"padding_top_unit\\":\\"%\\"}}},{\\"element_id\\":\\"m85g691\\",\\"cols\\":[{\\"element_id\\":\\"6oj4692\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"fancy-heading\\",\\"element_id\\":\\"2i6u506\\",\\"mod_settings\\":{\\"heading\\":\\"Why Us?\\",\\"text_alignment\\":\\"themify-text-center\\",\\"heading_tag\\":\\"h3\\",\\"padding_bottom\\":\\"30\\",\\"background_position\\":\\"center-bottom\\",\\"background_repeat\\":\\"no-repeat\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bg-title-arrow-down.png\\",\\"margin_bottom\\":\\"50\\",\\"font_size_unit\\":\\"px\\",\\"font_size\\":\\"40\\",\\"font_color_type\\":\\"font_color_solid\\"}},{\\"element_id\\":\\"m4ct469\\",\\"cols\\":[{\\"element_id\\":\\"i0my469\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"cndh185\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"style_image\\":\\"image-top\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/goal-icon.png\\",\\"margin_bottom\\":\\"30\\"}},{\\"mod_name\\":\\"icon\\",\\"element_id\\":\\"hx14793\\",\\"mod_settings\\":{\\"content_icon\\":[{\\"icon\\":\\"ti-arrow-right\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"Analyze Your Goal\\",\\"link_options\\":\\"regular\\"}],\\"icon_arrangement\\":\\"icon_horizontal\\",\\"icon_style\\":\\"circle\\",\\"icon_size\\":\\"normal\\",\\"font_color_type\\":\\"font_color_solid\\",\\"margin_bottom\\":\\"20\\",\\"font_color_icon\\":\\"#cc4415\\",\\"font_weight\\":\\"bold\\",\\"font_size_unit\\":\\"px\\",\\"font_size\\":\\"22\\",\\"font_color\\":\\"#303030\\"}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"y3ni450\\",\\"mod_settings\\":{\\"content_text\\":\\"<p>We assess and evaluate how to get you where you want to be - even if we have to start slow, we\\\\\\\\\\\'ll get there together.<\\\\/p>\\"}}],\\"styling\\":{\\"checkbox_padding_apply_all\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\"}},{\\"element_id\\":\\"5az4723\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"uch4650\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"style_image\\":\\"image-top\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/work-out-icon.png\\",\\"margin_bottom\\":\\"30\\"}},{\\"mod_name\\":\\"icon\\",\\"element_id\\":\\"qmtw49\\",\\"mod_settings\\":{\\"content_icon\\":[{\\"icon\\":\\"ti-arrow-right\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"Show Your Work\\",\\"link_options\\":\\"regular\\"}],\\"icon_arrangement\\":\\"icon_horizontal\\",\\"icon_style\\":\\"circle\\",\\"icon_size\\":\\"normal\\",\\"font_color_type\\":\\"font_color_solid\\",\\"margin_bottom\\":\\"20\\",\\"font_color_icon\\":\\"#bd441b\\",\\"font_weight\\":\\"bold\\",\\"font_size_unit\\":\\"px\\",\\"font_size\\":\\"22\\",\\"font_color\\":\\"#303030\\"}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"6uty245\\",\\"mod_settings\\":{\\"content_text\\":\\"<p>We\\\'ll guide you every step of the way, but show us your work. Let\\\'s see your dedication and commitment in action.<\\\\/p>\\"}}],\\"styling\\":{\\"checkbox_padding_apply_all\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\"}}]},{\\"element_id\\":\\"0cjo374\\",\\"cols\\":[{\\"element_id\\":\\"mhdq756\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"11gy757\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"style_image\\":\\"image-top\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/performance-icon.png\\",\\"margin_bottom\\":\\"30\\"}},{\\"mod_name\\":\\"icon\\",\\"element_id\\":\\"hj4l757\\",\\"mod_settings\\":{\\"content_icon\\":[{\\"icon\\":\\"ti-arrow-right\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"Improve Your Performance\\",\\"link_options\\":\\"regular\\"}],\\"icon_arrangement\\":\\"icon_horizontal\\",\\"icon_style\\":\\"circle\\",\\"icon_size\\":\\"normal\\",\\"font_color_type\\":\\"font_color_solid\\",\\"margin_bottom\\":\\"20\\",\\"font_color_icon\\":\\"#bd441b\\",\\"font_weight\\":\\"bold\\",\\"font_size_unit\\":\\"px\\",\\"font_size\\":\\"22\\",\\"font_color\\":\\"#303030\\"}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"6huq758\\",\\"mod_settings\\":{\\"content_text\\":\\"<p>Let\\\'s take a step back, re-assess, analyze, and re-commit to a the same or revised goal.<\\\\/p>\\"}}],\\"styling\\":{\\"checkbox_padding_apply_all\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\"}},{\\"element_id\\":\\"x8ci375\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"2g9r376\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"style_image\\":\\"image-top\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/achievement-icon.png\\",\\"margin_bottom\\":\\"30\\"}},{\\"mod_name\\":\\"icon\\",\\"element_id\\":\\"6mdv376\\",\\"mod_settings\\":{\\"content_icon\\":[{\\"icon\\":\\"ti-arrow-right\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"Achieve Your Destiny\\",\\"link_options\\":\\"regular\\"}],\\"icon_arrangement\\":\\"icon_horizontal\\",\\"icon_style\\":\\"circle\\",\\"icon_size\\":\\"normal\\",\\"font_color_type\\":\\"font_color_solid\\",\\"margin_bottom\\":\\"20\\",\\"font_color_icon\\":\\"#bd441b\\",\\"font_weight\\":\\"bold\\",\\"font_size_unit\\":\\"px\\",\\"font_size\\":\\"22\\",\\"font_color\\":\\"#303030\\"}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"twui376\\",\\"mod_settings\\":{\\"content_text\\":\\"<p>After all the hard work, sweat, and blood, feel the amazing and sweet feeling of victory. You did it!<\\\\/p>\\"}}],\\"styling\\":{\\"checkbox_padding_apply_all\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\"}}],\\"styling\\":{\\"padding_top\\":\\"0\\"}}]}],\\"styling\\":{\\"background_color\\":\\"#ecf3fd\\",\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"padding_bottom\\":\\"89\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_color\\":\\"#1a3068\\",\\"bottom-frame_layout\\":\\"slant1\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"1\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"146\\",\\"padding_bottom_unit\\":\\"px\\"}}},{\\"element_id\\":\\"dp1k939\\",\\"cols\\":[{\\"element_id\\":\\"90xc941\\",\\"grid_class\\":\\"col6-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"izo1956\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/adidas-logo.png\\",\\"style_image\\":\\"image-center\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"15\\",\\"padding_top\\":\\"15\\"}}}],\\"styling\\":{\\"breakpoint_tablet\\":{\\"padding_bottom\\":\\"10\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"10\\"}}},{\\"element_id\\":\\"sasa435\\",\\"grid_class\\":\\"col6-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"y8qb437\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/nike-logo.png\\",\\"style_image\\":\\"image-center\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"15\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"15\\"}}}],\\"styling\\":{\\"breakpoint_tablet\\":{\\"padding_bottom\\":\\"10\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"10\\"}}},{\\"element_id\\":\\"8puq491\\",\\"grid_class\\":\\"col6-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"ud5z491\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/reebok.png\\",\\"style_image\\":\\"image-center\\",\\"padding_top_unit\\":\\"%\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"15\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"15\\"}}}],\\"styling\\":{\\"breakpoint_tablet\\":{\\"padding_bottom\\":\\"10\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"10\\"}}},{\\"element_id\\":\\"o2si610\\",\\"grid_class\\":\\"col6-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"4fwt610\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/converse-logo.png\\",\\"style_image\\":\\"image-center\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"15\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"15\\"}}}],\\"styling\\":{\\"breakpoint_tablet\\":{\\"padding_bottom\\":\\"10\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"10\\"}}},{\\"element_id\\":\\"kdg4507\\",\\"grid_class\\":\\"col6-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"mbyt508\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/under-armour.png\\",\\"style_image\\":\\"image-center\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"15\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"15\\"}}}],\\"styling\\":{\\"breakpoint_tablet\\":{\\"padding_bottom\\":\\"10\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"10\\"}}},{\\"element_id\\":\\"d2e1331\\",\\"grid_class\\":\\"col6-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"2wkg331\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/puma.png\\",\\"style_image\\":\\"image-center\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"15\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"15\\"}}}],\\"styling\\":{\\"breakpoint_tablet\\":{\\"padding_bottom\\":\\"10\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"10\\"}}}],\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"background_color\\":\\"1a3068\\",\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"row_width\\":\\"fullwidth-content\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"3\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\",\\"row_anchor\\":\\"apparel\\"}},{\\"element_id\\":\\"3w93906\\",\\"cols\\":[{\\"element_id\\":\\"daoo908\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"fancy-heading\\",\\"element_id\\":\\"gq3u822\\",\\"mod_settings\\":{\\"heading\\":\\"Our Trainers\\",\\"sub_heading\\":\\"professionally trained \\",\\"text_alignment\\":\\"themify-text-center\\",\\"heading_tag\\":\\"h3\\",\\"padding_bottom\\":\\"30\\",\\"background_position\\":\\"center-bottom\\",\\"background_repeat\\":\\"no-repeat\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bg-title-arrow-down.png\\",\\"margin_bottom\\":\\"50\\",\\"font_color_subheading\\":\\"#ffffff\\",\\"font_color_type_subheading\\":\\"font_color_subheading_solid\\",\\"font_color\\":\\"#ffffff\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"font_size_subheading_unit\\":\\"px\\",\\"font_size_subheading\\":\\"26\\",\\"font_color_subheading\\":\\"#ffffff\\",\\"font_color_type_subheading\\":\\"font_color_subheading_solid\\"},\\"breakpoint_tablet\\":{\\"font_size_subheading_unit\\":\\"px\\",\\"font_size_subheading\\":\\"28\\",\\"font_color_subheading\\":\\"#ffffff\\",\\"font_color_type_subheading\\":\\"font_color_subheading_solid\\"}}},{\\"element_id\\":\\"nqad540\\",\\"cols\\":[{\\"element_id\\":\\"h5up541\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"my7i834\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"height_image\\":\\"260\\",\\"width_image\\":\\"260\\",\\"appearance_image\\":\\"rounded|circle\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bernard.jpg\\",\\"caption_on_overlay\\":\\"yes\\",\\"style_image\\":\\"image-full-overlay\\",\\"caption_image\\":\\"CrossFit Coach\\",\\"title_image\\":\\"Benji Jr.\\",\\"margin_bottom\\":\\"40\\",\\"i_t_b-type\\":\\"all\\",\\"i_t_b_top_style\\":\\"none\\",\\"border_top_width\\":\\"5\\",\\"border_top_color\\":\\"#bd441b\\",\\"border-type\\":\\"all\\",\\"background_position\\":\\"left-top\\",\\"background_repeat\\":\\"repeat\\",\\"font_color_type\\":\\"font_color_solid\\"}}],\\"styling\\":{\\"text_align\\":\\"center\\"}},{\\"element_id\\":\\"r8ha541\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"inx4570\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"height_image\\":\\"260\\",\\"width_image\\":\\"260\\",\\"appearance_image\\":\\"rounded|circle\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/monica-rodriguez.png\\",\\"caption_on_overlay\\":\\"yes\\",\\"style_image\\":\\"image-full-overlay\\",\\"caption_image\\":\\"Zumba Coach\\",\\"title_image\\":\\"Jessica Kim\\",\\"margin_bottom\\":\\"40\\",\\"i_t_b-type\\":\\"all\\",\\"i_t_b_top_style\\":\\"none\\",\\"border_top_width\\":\\"5\\",\\"border_top_color\\":\\"#bd441b\\",\\"border-type\\":\\"all\\",\\"background_position\\":\\"left-top\\",\\"background_repeat\\":\\"repeat\\",\\"font_color_type\\":\\"font_color_solid\\"}}],\\"styling\\":{\\"text_align\\":\\"center\\"}},{\\"element_id\\":\\"sew6541\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"c09j171\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"height_image\\":\\"260\\",\\"width_image\\":\\"260\\",\\"appearance_image\\":\\"rounded|circle\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/jessica.jpg\\",\\"caption_on_overlay\\":\\"yes\\",\\"style_image\\":\\"image-full-overlay\\",\\"caption_image\\":\\"Booty Boot Camp\\",\\"title_image\\":\\"Shania Dee\\",\\"margin_bottom\\":\\"40\\",\\"i_t_b-type\\":\\"all\\",\\"i_t_b_top_style\\":\\"none\\",\\"border_top_width\\":\\"5\\",\\"border_top_color\\":\\"#bd441b\\",\\"border-type\\":\\"all\\",\\"background_position\\":\\"left-top\\",\\"background_repeat\\":\\"repeat\\"}}],\\"styling\\":{\\"text_align\\":\\"center\\"}}],\\"styling\\":{\\"margin_bottom\\":\\"40\\"}},{\\"element_id\\":\\"zqvr564\\",\\"cols\\":[{\\"element_id\\":\\"u3zr566\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"pgin523\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"height_image\\":\\"260\\",\\"width_image\\":\\"260\\",\\"appearance_image\\":\\"rounded|circle\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/robert-moore.jpg\\",\\"caption_on_overlay\\":\\"yes\\",\\"style_image\\":\\"image-full-overlay\\",\\"caption_image\\":\\"Muay Thai\\",\\"title_image\\":\\"Robert Moore\\",\\"margin_bottom\\":\\"40\\",\\"i_t_b-type\\":\\"all\\",\\"i_t_b_top_style\\":\\"none\\",\\"border_top_width\\":\\"5\\",\\"border_top_color\\":\\"#bd441b\\",\\"border-type\\":\\"all\\",\\"background_position\\":\\"left-top\\",\\"background_repeat\\":\\"repeat\\"}}],\\"styling\\":{\\"text_align\\":\\"center\\"}},{\\"element_id\\":\\"ycz5567\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"3gk9564\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"height_image\\":\\"260\\",\\"width_image\\":\\"260\\",\\"appearance_image\\":\\"rounded|circle\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/michelle-kim.jpg\\",\\"caption_on_overlay\\":\\"yes\\",\\"style_image\\":\\"image-full-overlay\\",\\"caption_image\\":\\"Pilates\\",\\"title_image\\":\\"Michelina Jung\\",\\"margin_bottom\\":\\"40\\",\\"i_t_b-type\\":\\"all\\",\\"i_t_b_top_style\\":\\"none\\",\\"border_top_width\\":\\"5\\",\\"border_top_color\\":\\"#bd441b\\",\\"border-type\\":\\"all\\",\\"background_position\\":\\"left-top\\",\\"background_repeat\\":\\"repeat\\"}}],\\"styling\\":{\\"text_align\\":\\"center\\"}},{\\"element_id\\":\\"xp48567\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"9ko7445\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"height_image\\":\\"260\\",\\"width_image\\":\\"260\\",\\"appearance_image\\":\\"rounded|circle\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/michael.jpg\\",\\"caption_on_overlay\\":\\"yes\\",\\"style_image\\":\\"image-full-overlay\\",\\"caption_image\\":\\"Spinning\\",\\"title_image\\":\\"Michael Okue\\",\\"margin_bottom\\":\\"40\\",\\"i_t_b-type\\":\\"all\\",\\"i_t_b_top_style\\":\\"none\\",\\"border_top_width\\":\\"5\\",\\"border_top_color\\":\\"#bd441b\\",\\"border-type\\":\\"all\\",\\"background_position\\":\\"left-top\\",\\"background_repeat\\":\\"repeat\\"}}],\\"styling\\":{\\"text_align\\":\\"center\\"}}],\\"styling\\":{\\"margin_bottom\\":\\"40\\"}}]}],\\"styling\\":{\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_color\\":\\"1a3068\\",\\"top-frame_layout\\":\\"slant2\\",\\"top-frame_type\\":\\"top-presets\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"8\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"12\\",\\"border_inner-type\\":\\"top\\",\\"background_position_inner\\":\\"center-center\\",\\"background_attachment_inner\\":\\"scroll\\",\\"background_repeat_inner\\":\\"repeat\\",\\"cover_color\\":\\"#273447_0.89\\",\\"cover_color-type\\":\\"color\\",\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/just-waves.png\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"row_anchor\\":\\"trainers\\",\\"breakpoint_mobile\\":{\\"padding_top\\":\\"32\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom\\":\\"15\\",\\"padding_bottom_unit\\":\\"%\\"},\\"breakpoint_tablet\\":{\\"padding_top\\":\\"20\\",\\"padding_top_unit\\":\\"%\\"}}}]',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 179,
  'post_date' => '2019-07-28 10:01:34',
  'post_date_gmt' => '2019-07-28 10:01:34',
  'post_content' => '<!-- wp:themify-builder/canvas /--><!--themify_builder_static--><h1>Classes &amp; Activities</h1> <h3>Monthly Pass = free unlimited classes.</h3> <h3>Or join one class, get a bonus class free. </h3>
<h4>Aug</h4> <h3>15</h3> <h4>2019</h4>
<h3>Yoga for Runners</h3> <p>Condition and strengthen your body for your best run yet.</p>
<a href="https://themify.me/" > Register </a>
<h3>Details</h3>
Date : 15 August 2019 Time : 08:00 am - 17:00pm Cost : $100 Event : Yoga/Running
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/yoga.jpg" alt="yoga" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/yoga.jpg 555w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/yoga-300x185.jpg 300w" sizes="(max-width: 555px) 100vw, 555px" />
<p>There are many variations of passages of Lorem Ipsum available, but the majority have suffered alteration in some form, by injected humour, or randomised words which don\'t look even slightly believable.</p> <p>If you are going to use a passage of Lorem Ipsum, you need to be sure there isn\'t anything embarrassing hidden in the middle of text. All the Lorem Ipsum generators on the Internet tend to repeat predefined chunks as necessary, making this the first true generator on the Internet. It uses a dictionary of over 200 Latin words combined.</p>
<h3>Our Highlight Reel<br/>snapshots of glory</h3>
<a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-stretching.jpg" data-zoom-config="600px|400px" > <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-stretching-275x275.jpg" width="275" height="275" title="Jogging " alt="Jogging " srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-stretching-275x275.jpg 275w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-stretching-150x150.jpg 150w" sizes="(max-width: 275px) 100vw, 275px" /> </a> <h3> <a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-stretching.jpg" data-zoom-config="600px|400px" > Jogging </a> </h3>
<a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-together.jpg" data-zoom-config="600px|400px" > <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-together-275x275.jpg" width="275" height="275" title="Jogging Activity" alt="Jogging Activity" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-together-275x275.jpg 275w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-together-150x150.jpg 150w" sizes="(max-width: 275px) 100vw, 275px" /> </a> <h3> <a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-together.jpg" data-zoom-config="600px|400px" > Jogging Activity </a> </h3>
<a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/ready-to-run.jpg" data-zoom-config="600px|400px" > <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/ready-to-run-275x275.jpg" width="275" height="275" title="Ready To Run" alt="Ready To Run" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/ready-to-run-275x275.jpg 275w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/ready-to-run-150x150.jpg 150w" sizes="(max-width: 275px) 100vw, 275px" /> </a> <h3> <a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/ready-to-run.jpg" data-zoom-config="600px|400px" > Ready To Run </a> </h3>
<a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/wild-running.jpg" data-zoom-config="600px|400px" > <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/wild-running-275x275.jpg" width="275" height="275" title="Running " alt="Running " srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/wild-running-275x275.jpg 275w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/wild-running-150x150.jpg 150w" sizes="(max-width: 275px) 100vw, 275px" /> </a> <h3> <a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/wild-running.jpg" data-zoom-config="600px|400px" > Running </a> </h3>
<a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/hiking-running.jpg" data-zoom-config="600px|400px" > <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/hiking-running-275x275.jpg" width="275" height="275" title="Mountain Edge" alt="Mountain Edge" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/hiking-running-275x275.jpg 275w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/hiking-running-150x150.jpg 150w" sizes="(max-width: 275px) 100vw, 275px" /> </a> <h3> <a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/hiking-running.jpg" data-zoom-config="600px|400px" > Mountain Edge </a> </h3>
<a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-morning.jpg" data-zoom-config="600px|400px" > <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-morning-275x275.jpg" width="275" height="275" title="Morning Jogging" alt="Morning Jogging" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-morning-275x275.jpg 275w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-morning-150x150.jpg 150w" sizes="(max-width: 275px) 100vw, 275px" /> </a> <h3> <a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/jogging-morning.jpg" data-zoom-config="600px|400px" > Morning Jogging </a> </h3>
<a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/wild-run.jpg" data-zoom-config="600px|400px" > <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/wild-run-275x275.jpg" width="275" height="275" title="Running Across Nature " alt="Running Across Nature " srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/wild-run-275x275.jpg 275w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/wild-run-150x150.jpg 150w" sizes="(max-width: 275px) 100vw, 275px" /> </a> <h3> <a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/wild-run.jpg" data-zoom-config="600px|400px" > Running Across Nature </a> </h3>
<a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/bay-marathon.jpg" data-zoom-config="600px|400px" > <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/bay-marathon-275x275.jpg" width="275" height="275" title="Marathon Bay 2018" alt="Marathon Bay 2018" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/bay-marathon-275x275.jpg 275w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/bay-marathon-150x150.jpg 150w" sizes="(max-width: 275px) 100vw, 275px" /> </a> <h3> <a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/bay-marathon.jpg" data-zoom-config="600px|400px" > Marathon Bay 2018 </a> </h3><!--/themify_builder_static-->',
  'post_title' => 'Activities',
  'post_excerpt' => '',
  'post_name' => 'activities',
  'post_modified' => '2020-02-19 07:31:38',
  'post_modified_gmt' => '2020-02-19 07:31:38',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?page_id=179',
  'menu_order' => 0,
  'post_type' => 'page',
  'meta_input' => 
  array (
    'page_layout' => 'sidebar-none',
    'content_width' => 'full_width',
    'hide_page_title' => 'yes',
    'section_scrolling_mobile' => 'on',
    'mobile_menu_styles' => 'default',
    'header_wrap' => 'transparent',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"pn1e619\\",\\"cols\\":[{\\"element_id\\":\\"t006620\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"w5wq620\\",\\"mod_settings\\":{\\"content_text\\":\\"<h1>Classes &amp; Activities<\\\\/h1>\\\\n<h3>Monthly Pass = free unlimited classes.<\\\\/h3>\\\\n<h3>Or join one class, get a bonus class free. <\\\\/h3>\\",\\"margin_bottom\\":\\"40\\",\\"breakpoint_mobile\\":{\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"18\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\"}}}],\\"styling\\":{\\"font_color\\":\\"#ffffff\\",\\"breakpoint_tablet\\":{\\"padding_left\\":\\"18\\",\\"padding_left_unit\\":\\"px\\"},\\"breakpoint_mobile\\":{\\"padding_left\\":\\"1\\",\\"padding_left_unit\\":\\"px\\"}}},{\\"element_id\\":\\"hgyo621\\",\\"grid_class\\":\\"col4-2\\"}],\\"col_tablet\\":\\"column4-3-4-1\\",\\"styling\\":{\\"row_width\\":\\"fullwidth\\",\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/activities-hero-banner-bg.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"cover_color\\":\\"#0e2259_0.53\\",\\"cover_color-type\\":\\"color\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom\\":\\"18\\",\\"padding_top\\":\\"18\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"bottom-frame_height\\":\\"30\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"slant2\\",\\"breakpoint_mobile\\":{\\"padding_top\\":\\"26\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom\\":\\"25\\",\\"padding_bottom_unit\\":\\"%\\"}}},{\\"element_id\\":\\"naov150\\",\\"cols\\":[{\\"element_id\\":\\"toui151\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"element_id\\":\\"z7hb649\\",\\"cols\\":[{\\"element_id\\":\\"4f7x650\\",\\"grid_class\\":\\"col4-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"jqfn595\\",\\"mod_settings\\":{\\"content_text\\":\\"<h4>Aug<\\\\/h4>\\\\n<h3>15<\\\\/h3>\\\\n<h4>2019<\\\\/h4>\\",\\"line_height_h4_unit\\":\\"em\\",\\"line_height_h4\\":\\"1\\",\\"font_color_type_h4\\":\\"font_color_h4_solid\\",\\"text_align\\":\\"center\\",\\"line_height_unit\\":\\"em\\",\\"font_color_type\\":\\"font_color_solid\\",\\"font_size_h4_unit\\":\\"px\\",\\"font_size_h4\\":\\"24\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"em\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"line_height_h2_unit\\":\\"em\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"h3_margin_bottom\\":\\"10\\",\\"line_height_h3\\":\\"1\\",\\"font_size_h3\\":\\"50\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"breakpoint_mobile\\":{\\"margin_bottom\\":\\"30\\"},\\"breakpoint_tablet\\":{\\"margin_bottom\\":\\"20\\"}}}]},{\\"element_id\\":\\"gz7i650\\",\\"grid_class\\":\\"col4-3\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"dh10171\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Yoga for Runners<\\\\/h3>\\\\n<p>Condition and strengthen your body for your best run yet.<\\\\/p>\\",\\"font_weight_h3\\":\\"bold\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"36\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"line_height_unit\\":\\"em\\",\\"line_height\\":\\"1.3\\",\\"font_size_unit\\":\\"px\\",\\"font_size\\":\\"26\\",\\"font_color_type\\":\\"font_color_solid\\",\\"line_height_h3_unit\\":\\"em\\",\\"line_height_h3\\":\\"1\\",\\"breakpoint_mobile\\":{\\"font_weight_h3\\":\\"bold\\",\\"line_height_h3_unit\\":\\"em\\",\\"line_height_h3\\":\\"1\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"30\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"line_height_unit\\":\\"em\\",\\"line_height\\":\\"1.3\\",\\"font_size_unit\\":\\"px\\",\\"font_size\\":\\"22\\",\\"font_color_type\\":\\"font_color_solid\\"},\\"breakpoint_tablet\\":{\\"font_weight_h3\\":\\"bold\\",\\"line_height_h3_unit\\":\\"em\\",\\"line_height_h3\\":\\"1\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"30\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"line_height_unit\\":\\"em\\",\\"line_height\\":\\"1.3\\",\\"font_size_unit\\":\\"px\\",\\"font_size\\":\\"22\\",\\"font_color_type\\":\\"font_color_solid\\"}}}]}],\\"col_tablet\\":\\"column-full\\"},{\\"element_id\\":\\"awgj662\\",\\"cols\\":[{\\"element_id\\":\\"c7l1663\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"t5dx841\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Register\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"icon\\":\\"ti-arrow-right\\",\\"icon_alignment\\":\\"right\\",\\"button_color_bg\\":\\"tb_default_color\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_shape\\":\\"squared\\",\\"buttons_size\\":\\"normal\\",\\"margin_bottom\\":\\"50\\",\\"margin_top\\":\\"40\\",\\"buttons_style\\":\\"solid\\",\\"link_color\\":\\"#ffffff\\"}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"0p9p399\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Details<\\\\/h3>\\",\\"font_color_type_h5\\":\\"font_color_h5_solid\\",\\"font_weight_h3\\":\\"bold\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"36\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_tablet\\":{\\"font_weight_h3\\":\\"bold\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"30\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_color_type\\":\\"font_color_solid\\"}}},{\\"mod_name\\":\\"icon\\",\\"element_id\\":\\"i6as107\\",\\"mod_settings\\":{\\"content_icon\\":[{\\"icon\\":\\"ti-calendar\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"Date : 15 August 2019\\",\\"link_options\\":\\"regular\\"},{\\"icon\\":\\"ti-time\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"Time : 08:00 am - 17:00pm\\",\\"link_options\\":\\"regular\\"},{\\"icon\\":\\"ti-money\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"Cost : $100\\",\\"link_options\\":\\"regular\\"},{\\"icon\\":\\"ti-flag\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"Event : Yoga\\\\/Running\\",\\"link_options\\":\\"regular\\"}],\\"icon_position\\":\\"icon_position_left\\",\\"icon_arrangement\\":\\"icon_vertical\\",\\"icon_style\\":\\"circle\\",\\"icon_size\\":\\"small\\",\\"m_i_bottom\\":\\"10\\",\\"m_i_top\\":\\"10\\",\\"checkbox_p_i_apply_all\\":\\"1\\",\\"font_color_icon\\":\\"#ffffff\\",\\"background_color_icon\\":\\"#9fa1a6\\",\\"border-type\\":\\"bottom\\",\\"text_transform\\":\\"uppercase\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"margin_bottom\\":\\"30\\"}}}]}],\\"styling\\":{\\"padding_left\\":\\"30\\",\\"breakpoint_mobile\\":{\\"padding_left\\":\\"0\\"}}}]},{\\"element_id\\":\\"48ei151\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"gczj823\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/yoga.jpg\\",\\"style_image\\":\\"image-top\\",\\"margin_bottom\\":\\"30\\",\\"custom_parallax_scroll_speed\\":\\"1\\"}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"iw1v688\\",\\"mod_settings\\":{\\"content_text\\":\\"<p>There are many variations of passages of Lorem Ipsum available, but the majority have suffered alteration in some form, by injected humour, or randomised words which don\\\'t look even slightly believable.<\\\\/p>\\\\n<p>If you are going to use a passage of Lorem Ipsum, you need to be sure there isn\\\'t anything embarrassing hidden in the middle of text. All the Lorem Ipsum generators on the Internet tend to repeat predefined chunks as necessary, making this the first true generator on the Internet. It uses a dictionary of over 200 Latin words combined.<\\\\/p>\\"}}]}],\\"col_tablet\\":\\"column4-2\\",\\"styling\\":{\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"8\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"8\\",\\"row_anchor\\":\\"events\\"}},{\\"element_id\\":\\"5wp4184\\",\\"cols\\":[{\\"element_id\\":\\"rb5j186\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"fancy-heading\\",\\"element_id\\":\\"joxf141\\",\\"mod_settings\\":{\\"heading\\":\\"Our Highlight Reel\\",\\"sub_heading\\":\\"snapshots of glory\\",\\"text_alignment\\":\\"themify-text-center\\",\\"heading_tag\\":\\"h3\\",\\"padding_bottom\\":\\"30\\",\\"background_position\\":\\"center-bottom\\",\\"background_repeat\\":\\"no-repeat\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bg-title-arrow-down.png\\",\\"margin_bottom\\":\\"60\\",\\"breakpoint_mobile\\":{\\"font_size_subheading_unit\\":\\"px\\",\\"font_size_subheading\\":\\"26\\",\\"font_color_type_subheading\\":\\"font_color_subheading_solid\\"},\\"breakpoint_tablet\\":{\\"font_size_subheading_unit\\":\\"px\\",\\"font_size_subheading\\":\\"28\\",\\"font_color_type_subheading\\":\\"font_color_subheading_solid\\"}}},{\\"element_id\\":\\"t2nt496\\",\\"cols\\":[{\\"element_id\\":\\"yfjh496\\",\\"grid_class\\":\\"col4-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"s5zi77\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"zoom\\",\\"param_image\\":\\"lightbox\\",\\"height_image\\":\\"275\\",\\"width_image\\":\\"275\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/jogging-stretching.jpg\\",\\"style_image\\":\\"image-full-overlay\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/jogging-stretching.jpg\\",\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_height\\":\\"400\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_width\\":\\"600\\",\\"title_image\\":\\"Jogging \\",\\"caption_on_overlay\\":\\"yes\\"}}],\\"styling\\":{\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"10\\"}}},{\\"element_id\\":\\"79fd414\\",\\"grid_class\\":\\"col4-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"q6nt415\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"zoom\\",\\"param_image\\":\\"lightbox\\",\\"height_image\\":\\"275\\",\\"width_image\\":\\"275\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/jogging-together.jpg\\",\\"caption_on_overlay\\":\\"yes\\",\\"style_image\\":\\"image-full-overlay\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/jogging-together.jpg\\",\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_height\\":\\"400\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_width\\":\\"600\\",\\"title_image\\":\\"Jogging Activity\\"}}],\\"styling\\":{\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"10\\"}}},{\\"element_id\\":\\"68lk131\\",\\"grid_class\\":\\"col4-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"qpv4132\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"zoom\\",\\"param_image\\":\\"lightbox\\",\\"height_image\\":\\"275\\",\\"width_image\\":\\"275\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/ready-to-run.jpg\\",\\"caption_on_overlay\\":\\"yes\\",\\"style_image\\":\\"image-full-overlay\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/ready-to-run.jpg\\",\\"title_image\\":\\"Ready To Run\\",\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_height\\":\\"400\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_width\\":\\"600\\"}}],\\"styling\\":{\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"40\\"}}},{\\"element_id\\":\\"rmwt194\\",\\"grid_class\\":\\"col4-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"b3r9194\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"zoom\\",\\"param_image\\":\\"lightbox\\",\\"height_image\\":\\"275\\",\\"width_image\\":\\"275\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/wild-running.jpg\\",\\"caption_on_overlay\\":\\"yes\\",\\"style_image\\":\\"image-full-overlay\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/wild-running.jpg\\",\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_height\\":\\"400\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_width\\":\\"600\\",\\"title_image\\":\\"Running \\"}}],\\"styling\\":{\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"40\\"}}}],\\"col_tablet\\":\\"column4-1\\",\\"col_mobile\\":\\"column4-2\\",\\"styling\\":{\\"margin_bottom\\":\\"40\\",\\"checkbox_padding_apply_all\\":\\"1\\"}},{\\"element_id\\":\\"xcu8336\\",\\"cols\\":[{\\"element_id\\":\\"r32q175\\",\\"grid_class\\":\\"col4-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"u9l1175\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"zoom\\",\\"param_image\\":\\"lightbox\\",\\"height_image\\":\\"275\\",\\"width_image\\":\\"275\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/hiking-running.jpg\\",\\"style_image\\":\\"image-full-overlay\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/hiking-running.jpg\\",\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_height\\":\\"400\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_width\\":\\"600\\",\\"caption_on_overlay\\":\\"yes\\",\\"title_image\\":\\"Mountain Edge\\"}}],\\"styling\\":{\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"10\\"}}},{\\"element_id\\":\\"wl0k628\\",\\"grid_class\\":\\"col4-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"bwan629\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"zoom\\",\\"param_image\\":\\"lightbox\\",\\"height_image\\":\\"275\\",\\"width_image\\":\\"275\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/jogging-morning.jpg\\",\\"caption_on_overlay\\":\\"yes\\",\\"style_image\\":\\"image-full-overlay\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/jogging-morning.jpg\\",\\"title_image\\":\\"Morning Jogging\\",\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height\\":\\"400\\",\\"lightbox_width\\":\\"600\\"}}],\\"styling\\":{\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"10\\"}}},{\\"element_id\\":\\"k2q8934\\",\\"grid_class\\":\\"col4-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"62on935\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"zoom\\",\\"param_image\\":\\"lightbox\\",\\"height_image\\":\\"275\\",\\"width_image\\":\\"275\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/wild-run.jpg\\",\\"caption_on_overlay\\":\\"yes\\",\\"style_image\\":\\"image-full-overlay\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/wild-run.jpg\\",\\"title_image\\":\\"Running Across Nature \\",\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_height\\":\\"400\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_width\\":\\"600\\"}}],\\"styling\\":{\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"10\\"}}},{\\"element_id\\":\\"8tyn902\\",\\"grid_class\\":\\"col4-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"6bbz903\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"zoom\\",\\"param_image\\":\\"lightbox\\",\\"height_image\\":\\"275\\",\\"width_image\\":\\"275\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bay-marathon.jpg\\",\\"caption_on_overlay\\":\\"yes\\",\\"style_image\\":\\"image-full-overlay\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bay-marathon.jpg\\",\\"title_image\\":\\"Marathon Bay 2018\\",\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_height\\":\\"400\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_width\\":\\"600\\"}}],\\"styling\\":{\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"10\\"}}}],\\"col_tablet\\":\\"column4-1\\",\\"col_mobile\\":\\"column4-2\\",\\"styling\\":{\\"margin_bottom\\":\\"40\\",\\"checkbox_padding_apply_all\\":\\"1\\"}}]}],\\"styling\\":{\\"padding_top\\":\\"146\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ffffff\\",\\"top-frame_layout\\":\\"slant2\\",\\"top-frame_type\\":\\"top-presets\\",\\"background_color\\":\\"#ecf3fd\\",\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"8\\",\\"row_anchor\\":\\"gallery\\",\\"breakpoint_mobile\\":{\\"padding_top\\":\\"173\\",\\"padding_top_unit\\":\\"px\\"}}}]',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 176,
  'post_date' => '2019-07-28 10:01:14',
  'post_date_gmt' => '2019-07-28 10:01:14',
  'post_content' => '<!-- wp:themify-builder/canvas /--><!--themify_builder_static--><h1>Get Your 3 Free Sessions!</h1> <h3>Set yourself up for success.</h3>
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/yoga-1.jpg" alt="yoga" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/yoga-1.jpg 700w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/yoga-1-300x250.jpg 300w" sizes="(max-width: 700px) 100vw, 700px" />
<h3>We make sign ups easy!</h3> <p>If you\'ve never visited us before or if it\'s been a while, well we\'ve got a treat for you! By signing up today using our contact form below, you\'ll get a free pass for 3 classes. </p> <p>Or if you\'re thinking about joining a community of runners, pilate lovers, CrossFitters, or yogis, we can help connect you. Contact us today to get started.</p>
<a href="https://themify.me/demo/themes/ultra-trainers/contact/#form" > Book Now </a>
<h3>Drop by!</h3> <p>Whatever stage you\'re at, however you feel, our gym and special trainers have your back. Come and join us for a free class!</p>
84 Dundas St W, Toronto, ON, Canada 343 Bathurst St, Toronto, ON, Canada 101 Carr St, Toronto, ON, Canada
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/map-small-630x500.jpg" width="630" height="500" alt="map-small" />
<h3>Contact Us!<br/>book a class, a personal trainer, or 3 free sessions</h3>
<form action="https://themify.me/demo/themes/ultra-trainers/wp-admin/admin-ajax.php" id="contact-0--form" method="post"> <label for="contact-0--contact-name">Name *</label> <input type="text" name="contact-name" placeholder="" id="contact-0--contact-name" value="" required/> <label for="contact-0--contact-email">Email *</label> <input type="text" name="contact-email" placeholder="" id="contact-0--contact-email" value="" required /> <label for="contact-0--contact-subject">Subject *</label> <input type="text" name="contact-subject" placeholder="" id="contact-0--contact-subject" value="" required /> <label for="contact-0--contact-message">Message *</label> <textarea name="contact-message" placeholder="" id="contact-0--contact-message" rows="8" cols="45" required></textarea> <button type="submit"> Send </button> </form><!--/themify_builder_static-->',
  'post_title' => 'Contact',
  'post_excerpt' => '',
  'post_name' => 'contact',
  'post_modified' => '2019-09-19 15:40:00',
  'post_modified_gmt' => '2019-09-19 15:40:00',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?page_id=176',
  'menu_order' => 0,
  'post_type' => 'page',
  'meta_input' => 
  array (
    'page_layout' => 'sidebar-none',
    'content_width' => 'full_width',
    'hide_page_title' => 'yes',
    'section_scrolling_mobile' => 'on',
    'mobile_menu_styles' => 'default',
    'header_wrap' => 'transparent',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"9xkm62\\",\\"cols\\":[{\\"element_id\\":\\"dxak63\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"9sn964\\",\\"mod_settings\\":{\\"content_text\\":\\"<h1>Get Your 3 Free Sessions!<\\\\/h1>\\\\n<h3>Set yourself up for success.<\\\\/h3>\\",\\"margin_bottom\\":\\"40\\",\\"breakpoint_mobile\\":{\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"18\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\"}}}],\\"styling\\":{\\"font_color\\":\\"#ffffff\\"}},{\\"element_id\\":\\"lqn865\\",\\"grid_class\\":\\"col4-2\\"}],\\"styling\\":{\\"row_width\\":\\"fullwidth\\",\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https://themify.me/demo/themes/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/activities.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"cover_color\\":\\"#0e2259_0.53\\",\\"cover_color-type\\":\\"color\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom\\":\\"18\\",\\"padding_top\\":\\"18\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"bottom-frame_height\\":\\"30\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"slant2\\",\\"breakpoint_mobile\\":{\\"padding_top\\":\\"28\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom\\":\\"33\\",\\"padding_bottom_unit\\":\\"%\\"}}},{\\"element_id\\":\\"vno7587\\",\\"cols\\":[{\\"element_id\\":\\"9vhx590\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"element_id\\":\\"l3y9619\\",\\"cols\\":[{\\"element_id\\":\\"88mj620\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"z1yb734\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"url_image\\":\\"https://themify.me/demo/themes/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/yoga-1.jpg\\",\\"style_image\\":\\"image-top\\",\\"custom_parallax_scroll_speed\\":\\"2\\"}}],\\"styling\\":{\\"breakpoint_mobile\\":{\\"margin-bottom\\":\\"20\\",\\"margin-bottom_unit\\":\\"px\\"}}},{\\"element_id\\":\\"nwgy620\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"97ow961\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>We make sign ups easy!<\\\\/h3>\\\\n<p>If you\\\\\\\\\\\'ve never visited us before or if it\\\\\\\\\\\'s been a while, well we\\\\\\\\\\\'ve got a treat for you! By signing up today using our contact form below, you\\\\\\\\\\\'ll get a free pass for 3 classes. <\\\\/p>\\\\n<p>Or if you\\\\\\\\\\\'re thinking about joining a community of runners, pilate lovers, CrossFitters, or yogis, we can help connect you. Contact us today to get started.<\\\\/p>\\",\\"margin_bottom\\":\\"11\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"36\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"margin_bottom_unit\\":\\"px\\",\\"breakpoint_tablet\\":{\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"30\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\"}}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"eben18\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Book Now\\",\\"link\\":\\"https://themify.me/demo/themes/ultra-trainers\\\\/contact\\\\/#form\\",\\"link_options\\":\\"regular\\",\\"icon\\":\\"ti-arrow-right\\",\\"icon_alignment\\":\\"right\\",\\"button_color_bg\\":\\"tb_default_color\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_shape\\":\\"squared\\",\\"buttons_size\\":\\"normal\\"}}],\\"styling\\":{\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"3\\",\\"breakpoint_mobile\\":{\\"padding_top\\":\\"0\\"}}}],\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column4-2\\"}]}],\\"styling\\":{\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"0\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"2\\",\\"breakpoint_mobile\\":{\\"padding_top\\":\\"6\\",\\"padding_top_unit\\":\\"%\\"}}},{\\"element_id\\":\\"0pp5943\\",\\"cols\\":[{\\"element_id\\":\\"9109945\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"ki65416\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Drop by!<\\\\/h3>\\\\n<p>Whatever stage you\\\\\\\\\\\'re at, however you feel, our gym and special trainers have your back. Come and join us for a free class!<\\\\/p>\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"36\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"breakpoint_tablet\\":{\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"30\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\"}}},{\\"mod_name\\":\\"icon\\",\\"element_id\\":\\"53s3209\\",\\"mod_settings\\":{\\"content_icon\\":[{\\"icon\\":\\"ti-location-pin\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\" 84 Dundas St W, Toronto, ON, Canada\\",\\"link_options\\":\\"regular\\"},{\\"icon\\":\\"ti-location-pin\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"343 Bathurst St, Toronto, ON, Canada\\",\\"link_options\\":\\"regular\\"},{\\"icon\\":\\"ti-location-pin\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"101 Carr St, Toronto, ON, Canada\\",\\"link_options\\":\\"regular\\"}],\\"icon_position\\":\\"icon_position_left\\",\\"icon_arrangement\\":\\"icon_vertical\\",\\"icon_style\\":\\"circle\\",\\"icon_size\\":\\"large\\"}}],\\"styling\\":{\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"3\\",\\"breakpoint_mobile\\":{\\"margin-bottom\\":\\"38\\",\\"margin-bottom_unit\\":\\"px\\"}}},{\\"element_id\\":\\"rooe734\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"6pof404\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"style_image\\":\\"image-top\\",\\"height_image\\":\\"500\\",\\"width_image\\":\\"630\\",\\"url_image\\":\\"https://themify.me/demo/themes/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/map-small.jpg\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column4-2\\",\\"styling\\":{\\"padding_top_unit\\":\\"%\\",\\"breakpoint_mobile\\":{\\"padding_top\\":\\"10\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom\\":\\"10\\",\\"padding_bottom_unit\\":\\"%\\"},\\"breakpoint_tablet\\":{\\"padding_top\\":\\"5\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom\\":\\"9\\",\\"padding_bottom_unit\\":\\"%\\"}}},{\\"element_id\\":\\"s4i0808\\",\\"cols\\":[{\\"element_id\\":\\"6m1y808\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"fancy-heading\\",\\"element_id\\":\\"ethf872\\",\\"mod_settings\\":{\\"heading\\":\\"Contact Us!\\",\\"sub_heading\\":\\"book a class, a personal trainer, or 3 free sessions\\",\\"text_alignment\\":\\"themify-text-center\\",\\"heading_tag\\":\\"h3\\",\\"padding_bottom\\":\\"30\\",\\"background_position\\":\\"center-bottom\\",\\"background_repeat\\":\\"no-repeat\\",\\"background_image\\":\\"https://themify.me/demo/themes/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bg-title-arrow-down.png\\",\\"margin_bottom\\":\\"60\\",\\"font_color_subheading\\":\\"#ffffff\\",\\"font_color_type_subheading\\":\\"font_color_subheading_solid\\",\\"font_color\\":\\"#ffffff\\",\\"font_color_type\\":\\"font_color_solid\\"}},{\\"mod_name\\":\\"contact\\",\\"element_id\\":\\"35nt665\\",\\"mod_settings\\":{\\"field_name_label\\":\\"Name\\",\\"field_email_label\\":\\"Email\\",\\"field_subject_label\\":\\"Subject\\",\\"field_message_label\\":\\"Message\\",\\"field_sendcopy_label\\":\\"Send a copy to myself\\",\\"field_send_label\\":\\"Send\\",\\"gdpr_label\\":\\"I consent to my submitted data being collected and stored\\",\\"field_name_require\\":\\"yes\\",\\"field_email_require\\":\\"yes\\",\\"field_name_active\\":\\"yes\\",\\"field_email_active\\":\\"yes\\",\\"field_subject_require\\":\\"yes\\",\\"field_message_active\\":\\"yes\\",\\"field_send_align\\":\\"center\\",\\"field_extra\\":\\"{ \\\\\\\\\\\\\\"fields\\\\\\\\\\\\\\": [] }\\",\\"field_order\\":\\"{}\\",\\"contact_sent_from\\":\\"enable\\",\\"layout_contact\\":\\"style2\\",\\"field_subject_active\\":\\"yes\\",\\"font_color_labels\\":\\"#ffffff\\",\\"font_color_type\\":\\"font_color_solid\\",\\"field_sendcopy_active\\":\\"\\",\\"field_captcha_active\\":\\"\\",\\"post_author\\":\\"\\",\\"send_to_admins\\":\\"true\\"}}],\\"styling\\":{\\"padding_right_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"8\\",\\"padding_right\\":\\"8\\"}}],\\"styling\\":{\\"cover_color\\":\\"#0d2059_0.71\\",\\"cover_color-type\\":\\"color\\",\\"background_position\\":\\"center-top\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https://themify.me/demo/themes/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bg-contact.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"8\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"8\\",\\"row_width\\":\\"fullwidth\\",\\"row_anchor\\":\\"form\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"15\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top\\":\\"15\\",\\"padding_top_unit\\":\\"%\\"},\\"breakpoint_tablet\\":{\\"padding_bottom\\":\\"11\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top\\":\\"11\\",\\"padding_top_unit\\":\\"%\\"}}}]',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 7,
  'post_date' => '2019-07-13 03:37:59',
  'post_date_gmt' => '2019-07-13 03:37:59',
  'post_content' => '<!-- wp:themify-builder/canvas /--><!--themify_builder_static--><h1>Ready, Set, Go!</h1> <h3>Get your first 3 workout sessions from us,</h3> <h3>totally free.</h3>
<a href="https://themify.me/demo/themes/ultra-trainers/activities/" > Get Started </a>
<img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/push-up-1.jpg" alt="push-up" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/push-up-1.jpg 555w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/push-up-1-300x240.jpg 300w" sizes="(max-width: 555px) 100vw, 555px" />
<p>01</p>
<h3>Push Your Limits</h3> <p>Summer is an important season to keep your client sign-up flow growing. Keep your current and potential clients engaged, inspired, and in touch with all of the new workout activities you or your gym are offering. Use the Ultra Trainers skin to reach your audience today.</p>
<p>02</p>
<a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/heavy-weight-lift.jpg" > <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/heavy-weight-lift-555x493.jpg" width="555" alt="heavy-weight-lift" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/heavy-weight-lift.jpg 555w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/heavy-weight-lift-300x267.jpg 300w" sizes="(max-width: 555px) 100vw, 555px" /> </a>
<a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/athelete-trainer.jpg" > <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/athelete-trainer-555x370.jpg" width="555" alt="athelete-trainer" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/athelete-trainer-555x370.jpg 555w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/athelete-trainer-300x200.jpg 300w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/athelete-trainer.jpg 700w" sizes="(max-width: 555px) 100vw, 555px" /> </a>
<a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/personal-trainer.jpg" > <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/personal-trainer-555x370.jpg" width="555" alt="personal-trainer" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/personal-trainer-555x370.jpg 555w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/personal-trainer-300x200.jpg 300w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/personal-trainer.jpg 600w" sizes="(max-width: 555px) 100vw, 555px" /> </a>
<a href="https://themify.me/demo/themes/ultra-trainers/files/2019/07/leg-excersise.jpg" > <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/leg-excersise-555x493.jpg" width="555" alt="leg-excersise" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/leg-excersise.jpg 555w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/leg-excersise-300x267.jpg 300w" sizes="(max-width: 555px) 100vw, 555px" /> </a>
<h3>Our Power Trainers<br/>the dedication you need</h3>
<p>03</p>
<ul data-id="tb_qhy6775" data-visible="1" data-mob-visible="0" data-scroll="1" data-auto-scroll="off" data-speed="1" data-wrap="yes" data-arrow="yes" data-pagination="yes" data-effect="scroll" data-height="variable" data-pause-on-hover="resume" data-play-controller="no" data-horizontal="no" > 
 <li> <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/bernard-johnson-280x475.png" width="280" height="475" alt="Bernard Johnson" /> 
 <h3> Bernard Johnson </h3> Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Quis ipsum suspendisse ultrices gravida. Risus comv vcvv vmodo viverra maecenas accumsan lacus vel facilisis. </li> <li> <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/kimberly-mckinley-280x475.jpg" width="280" height="475" alt="Kimberly Mckinley" /> 
 <h3> Kimberly Mckinley </h3> Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Quis ipsum suspendisse ultrices gravida. Risus comv vcvv vmodo viverra maecenas accumsan lacus vel facilisis. </li> <li> <img src="https://themify.me/demo/themes/ultra-trainers/files/2019/07/arturo-dominic-280x475.jpg" width="280" height="475" alt="Arturo Dominic" srcset="https://themify.me/demo/themes/ultra-trainers/files/2019/07/arturo-dominic.jpg 280w, https://themify.me/demo/themes/ultra-trainers/files/2019/07/arturo-dominic-177x300.jpg 177w" sizes="(max-width: 280px) 100vw, 280px" /> 
 <h3> Arturo Dominic </h3> Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Quis ipsum suspendisse ultrices gravida. Risus comv vcvv vmodo viverra maecenas accumsan lacus vel facilisis. </li> </ul>
<h3>Programs<br/>build a stronger body today</h3>
<p>04</p>
<h3>CrossFit</h3> <p>Sunday <br />08am - 10:30am</p> <h4>Benji Jr.</h4>
<h3>Zumba</h3> <p>Monday<br />08am - 10:30am</p> <h4>Jessica  Kim</h4>
<h3>Muay Thai</h3> <p>Tuesday<br />08am - 10:30am</p> <h4>Robert Moore</h4>
<h3>Yoga</h3> <p>Wednesday<br />08am - 10:30am</p> <h4>Arunima Ali</h4>
<h3>Pilates</h3> <p>Thursday<br />08am - 10:30am</p> <h4>Michelina JuNG</h4>
<h3>Body Pump</h3> <p>Friday<br />08am - 10:30am</p> <h4>Arturo Dominic</h4>
<h3>Booty Boot Camp</h3> <p>Saturday<br />08am - 10:30am</p> <h4>Shania Dee</h4>
<h3>Boxing 101</h3> <p>Sunday <br />08am - 10:30am</p> <h4>Kenny Cheung</h4>
<h3>Barre Basics</h3> <p>Sunday<br />08am - 10:30am</p> <h4>Olga Kane</h4>
<h3>Spinning</h3> <p>Sunday <br />08am - 10:30am</p> <h4>Michael Okue</h4>
<a href="https://themify.me/demo/themes/ultra-trainers/contact/" > Book Now </a>
<h3>Our Locations<br/>map & addresses</h3>
84 Dundas St W, Toronto, ON, Canada 343 Bathurst St, Toronto, ON, Canada 101 Carr St, Toronto, ON, Canada<!--/themify_builder_static-->',
  'post_title' => 'Home',
  'post_excerpt' => '',
  'post_name' => 'home',
  'post_modified' => '2020-04-14 23:51:56',
  'post_modified_gmt' => '2020-04-14 23:51:56',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?page_id=7',
  'menu_order' => 0,
  'post_type' => 'page',
  'meta_input' => 
  array (
    'page_layout' => 'sidebar-none',
    'content_width' => 'full_width',
    'hide_page_title' => 'yes',
    'section_scrolling_mobile' => 'on',
    'mobile_menu_styles' => 'default',
    'header_wrap' => 'transparent',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"ccep912\\",\\"cols\\":[{\\"element_id\\":\\"j44y914\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"3ijc309\\",\\"mod_settings\\":{\\"content_text\\":\\"<h1>Ready, Set, Go!<\\\\/h1>\\\\n<h3>Get your first 3 workout sessions from us,<\\\\/h3>\\\\n<h3>totally free.<\\\\/h3>\\",\\"margin_bottom\\":\\"40\\"}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"rch2117\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Get Started\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/activities\\\\/\\",\\"link_options\\":\\"regular\\",\\"icon\\":\\"ti-arrow-right\\",\\"icon_alignment\\":\\"right\\",\\"button_color_bg\\":\\"tb_default_color\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_shape\\":\\"squared\\",\\"buttons_size\\":\\"large\\",\\"link_color\\":\\"#ffffff\\",\\"font_color_type\\":\\"font_color_solid\\"}}],\\"styling\\":{\\"font_color\\":\\"#ffffff\\"}},{\\"element_id\\":\\"zj43678\\",\\"grid_class\\":\\"col4-2\\"}],\\"styling\\":{\\"row_width\\":\\"fullwidth\\",\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bg-hero-banner-home.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"cover_color\\":\\"#0e2259_0.53\\",\\"cover_color-type\\":\\"color\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom\\":\\"18\\",\\"padding_top\\":\\"18\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"bottom-frame_height\\":\\"30\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"slant1\\",\\"breakpoint_mobile\\":{\\"padding_top\\":\\"35\\",\\"padding_top_unit\\":\\"%\\"}}},{\\"element_id\\":\\"ip9i616\\",\\"cols\\":[{\\"element_id\\":\\"idey617\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"3sx5618\\",\\"mod_settings\\":{\\"param_image\\":\\"regular\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/push-up-1.jpg\\",\\"style_image\\":\\"image-center\\",\\"custom_parallax_scroll_zindex\\":\\"2\\",\\"custom_parallax_scroll_speed\\":\\"1\\",\\"image_zoom_icon\\":false,\\"auto_fullwidth\\":false,\\"appearance_image\\":false,\\"caption_on_overlay\\":false}}],\\"styling\\":{\\"padding_right\\":\\"0\\",\\"breakpoint_tablet\\":{\\"padding_right\\":\\"0\\"}}},{\\"element_id\\":\\"20k3619\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"lyuk619\\",\\"mod_settings\\":{\\"content_text\\":\\"<p>01<\\\\/p>\\",\\"custom_parallax_scroll_speed\\":\\"1\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"p_margin_top_unit\\":\\"%\\",\\"p_margin_top\\":\\"-25\\",\\"font_weight\\":\\"bold\\",\\"text_align\\":\\"right\\",\\"line_height_unit\\":\\"em\\",\\"line_height\\":\\"1.2\\",\\"font_size_unit\\":\\"em\\",\\"font_size\\":\\"10\\",\\"font_gradient_color-gradient\\":\\"0% rgb(217, 88, 37)|100% rgb(173, 63, 16)\\",\\"font_color_type\\":\\"font_gradient_color_gradient\\",\\"background_position\\":\\"left-top\\",\\"background_repeat\\":\\"repeat\\",\\"font_style_regular\\":\\"normal\\",\\"font_family_w\\":\\"800\\",\\"font_family\\":\\"Open Sans\\"}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"tgla189\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Push Your Limits<\\\\/h3>\\\\n<p>Summer is an important season to keep your client sign-up flow growing. Keep your current and potential clients engaged, inspired, and in touch with all of the new workout activities you or your gym are offering. Use the Ultra Trainers skin to reach your audience today.<\\\\/p>\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"20\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"12\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"10\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"12\\",\\"background_position\\":\\"left-top\\",\\"background_repeat\\":\\"repeat\\",\\"background_color\\":\\"#ffffff\\",\\"margin_left_unit\\":\\"%\\",\\"margin_left\\":\\"-25\\",\\"margin_bottom_unit\\":\\"%\\",\\"margin_right_unit\\":\\"%\\",\\"margin_top_unit\\":\\"%\\",\\"margin_top\\":\\"22\\",\\"margin_bottom\\":\\"8\\",\\"margin_right\\":\\"18\\",\\"custom_parallax_scroll_speed\\":\\"2\\",\\"breakpoint_tablet\\":{\\"margin_bottom_unit\\":\\"%\\",\\"margin_right_unit\\":\\"%\\",\\"margin_top_unit\\":\\"%\\",\\"margin_left\\":\\"30\\",\\"margin_bottom\\":\\"8\\",\\"margin_right\\":\\"18\\",\\"margin_top\\":\\"22\\",\\"checkbox_padding_apply_all\\":\\"1\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_right_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"10\\",\\"padding_bottom\\":\\"12\\",\\"padding_right\\":\\"10\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"10\\"}}}],\\"styling\\":{\\"border_top_width\\":\\"5\\",\\"border_top_color\\":\\"#d95825\\",\\"border-type\\":\\"all\\",\\"margin-bottom_unit\\":\\"%\\",\\"margin-top_unit\\":\\"%\\",\\"margin-top\\":\\"6\\",\\"padding_bottom\\":\\"4\\",\\"padding_bottom_unit\\":\\"%\\",\\"breakpoint_tablet\\":{\\"margin-bottom_unit\\":\\"%\\",\\"margin-top_unit\\":\\"%\\",\\"margin-top\\":\\"6\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"4\\"}}}],\\"column_h\\":\\"1\\",\\"col_tablet\\":\\"column-full\\",\\"styling\\":{\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom\\":\\"18\\",\\"padding_top\\":\\"6\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bg-about-us-section.jpg\\",\\"row_width\\":\\"fullwidth\\"}},{\\"element_id\\":\\"ejd5194\\",\\"cols\\":[{\\"element_id\\":\\"vv38197\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"nlgs247\\",\\"mod_settings\\":{\\"content_text\\":\\"<p>02<\\\\/p>\\",\\"font_gradient_color_h1-gradient\\":\\"0% rgb(217, 88, 37)|100% rgb(173, 63, 16)\\",\\"custom_parallax_scroll_speed\\":\\"1\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"p_margin_top_unit\\":\\"%\\",\\"p_margin_top\\":\\"-25\\",\\"font_weight\\":\\"bold\\",\\"text_align\\":\\"left\\",\\"line_height_unit\\":\\"em\\",\\"line_height\\":\\"1\\",\\"font_size_unit\\":\\"em\\",\\"font_size\\":\\"10\\",\\"font_color_type\\":\\"font_gradient_color_gradient\\",\\"margin_left_unit\\":\\"%\\",\\"margin_left\\":\\"-30\\",\\"font_gradient_color-gradient\\":\\"0% rgb(217, 88, 37)|100% rgb(173, 63, 16)\\",\\"margin_bottom_unit\\":\\"%\\",\\"margin_bottom\\":\\"-50\\",\\"margin_top_unit\\":\\"%\\",\\"font_style_regular\\":\\"normal\\",\\"font_family_w\\":\\"800\\",\\"font_family\\":\\"Open Sans\\"}},{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"e2c6278\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"zoom\\",\\"param_image\\":\\"lightbox\\",\\"width_image\\":\\"555\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/heavy-weight-lift.jpg\\",\\"style_image\\":\\"image-center\\",\\"margin_bottom\\":\\"40\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/heavy-weight-lift.jpg\\"}},{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"8apv272\\",\\"mod_settings\\":{\\"param_image\\":\\"lightbox\\",\\"width_image\\":\\"555\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/athelete-trainer.jpg\\",\\"style_image\\":\\"image-center\\",\\"margin_bottom\\":\\"40\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/athelete-trainer.jpg\\",\\"image_zoom_icon\\":\\"zoom\\"}}]},{\\"element_id\\":\\"o8jm416\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"grlv177\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"zoom\\",\\"param_image\\":\\"lightbox\\",\\"width_image\\":\\"555\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/personal-trainer.jpg\\",\\"style_image\\":\\"image-center\\",\\"margin_bottom\\":\\"40\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/personal-trainer.jpg\\"}},{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"jrpq646\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"zoom\\",\\"param_image\\":\\"lightbox\\",\\"width_image\\":\\"555\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/leg-excersise.jpg\\",\\"style_image\\":\\"image-center\\",\\"margin_bottom\\":\\"40\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/leg-excersise.jpg\\"}}],\\"styling\\":{\\"margin-top_unit\\":\\"%\\",\\"margin-top\\":\\"12\\",\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\"}}],\\"styling\\":{\\"row_width\\":\\"fullwidth\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"0\\"}},{\\"element_id\\":\\"dcqv122\\",\\"cols\\":[{\\"element_id\\":\\"bqo9124\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"fancy-heading\\",\\"element_id\\":\\"tn27590\\",\\"mod_settings\\":{\\"heading\\":\\"Our Power Trainers\\",\\"sub_heading\\":\\"the dedication you need\\",\\"text_alignment\\":\\"themify-text-center\\",\\"heading_tag\\":\\"h3\\",\\"padding_bottom\\":\\"30\\",\\"background_position\\":\\"center-bottom\\",\\"background_repeat\\":\\"no-repeat\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bg-title-arrow-down.png\\",\\"margin_bottom_unit\\":\\"%\\",\\"margin_bottom\\":\\"-8\\",\\"breakpoint_mobile\\":{\\"font_size_subheading_unit\\":\\"px\\",\\"font_size_subheading\\":\\"26\\",\\"font_color_type_subheading\\":\\"font_color_subheading_solid\\"}}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"3ae0424\\",\\"mod_settings\\":{\\"content_text\\":\\"<p>03<\\\\/p>\\",\\"custom_parallax_scroll_speed\\":\\"1\\",\\"background_position\\":\\"left-top\\",\\"background_repeat\\":\\"repeat\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"margin_bottom_unit\\":\\"%\\",\\"margin_bottom\\":\\"-20\\",\\"font_weight\\":\\"bold\\",\\"font_style_regular\\":\\"normal\\",\\"text_align\\":\\"right\\",\\"font_size_unit\\":\\"em\\",\\"font_size\\":\\"10\\",\\"font_gradient_color-gradient\\":\\"0% rgb(217, 88, 37)|100% rgb(173, 63, 16)\\",\\"font_color_type\\":\\"font_gradient_color_gradient\\",\\"font_family_w\\":\\"800\\",\\"font_family\\":\\"Open Sans\\",\\"line_height_unit\\":\\"em\\",\\"line_height\\":\\"1.2\\"}},{\\"mod_name\\":\\"slider\\",\\"element_id\\":\\"qhy6775\\",\\"mod_settings\\":{\\"posts_per_page_slider\\":\\"4\\",\\"display_slider\\":\\"none\\",\\"img_w_slider\\":\\"280\\",\\"img_h_slider\\":\\"475\\",\\"horizontal\\":\\"no\\",\\"visible_opt_slider\\":\\"1\\",\\"pause_on_hover_slider\\":\\"resume\\",\\"show_arrow_slider\\":\\"yes\\",\\"show_nav_slider\\":\\"yes\\",\\"wrap_slider\\":\\"yes\\",\\"auto_scroll_opt_slider\\":\\"off\\",\\"post_type\\":\\"post\\",\\"height_slider\\":\\"variable\\",\\"show_arrow_buttons_vertical\\":\\"vertical\\",\\"effect_slider\\":\\"scroll\\",\\"speed_opt_slider\\":\\"normal\\",\\"scroll_opt_slider\\":\\"1\\",\\"mob_visible_opt_slider\\":\\"0\\",\\"layout_slider\\":\\"slider-agency\\",\\"img_content_slider\\":[{\\"img_url_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bernard-johnson.png\\",\\"img_title_slider\\":\\"Bernard Johnson\\",\\"img_caption_slider\\":\\"Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Quis ipsum suspendisse ultrices gravida. Risus comv vcvv                                                                                                                                                                            vmodo viverra maecenas accumsan lacus vel facilisis. \\"},{\\"img_url_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/kimberly-mckinley.jpg\\",\\"img_title_slider\\":\\"Kimberly Mckinley\\",\\"img_caption_slider\\":\\"Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Quis ipsum suspendisse ultrices gravida. Risus comv vcvv                                                                                                                                                                            vmodo viverra maecenas accumsan lacus vel facilisis. \\"},{\\"img_url_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/arturo-dominic.jpg\\",\\"img_title_slider\\":\\"Arturo Dominic\\",\\"img_caption_slider\\":\\"Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Quis ipsum suspendisse ultrices gravida. Risus comv vcvv                                                                                                                                                                            vmodo viverra maecenas accumsan lacus vel facilisis. \\"}],\\"open_link_new_tab_slider\\":\\"no\\",\\"unlink_feat_img_slider\\":\\"no\\",\\"hide_feat_img_slider\\":\\"no\\",\\"unlink_post_title_slider\\":\\"no\\",\\"hide_post_title_slider\\":\\"no\\",\\"orderby_slider\\":\\"date\\",\\"order_slider\\":\\"desc\\",\\"blog_category_slider\\":\\"null|single\\",\\"taxonomy\\":\\"category\\",\\"layout_display_slider\\":\\"image\\",\\"margin_bottom\\":\\"40\\",\\"margin_top\\":\\"40\\",\\"breakpoint_mobile\\":{\\"m_title_top\\":\\"30\\"}}}],\\"styling\\":{\\"text_align\\":\\"center\\"}}],\\"styling\\":{\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"2\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"2\\",\\"row_width\\":\\"fullwidth\\",\\"margin-bottom\\":\\"-6\\",\\"margin-bottom_unit\\":\\"px\\"}},{\\"element_id\\":\\"li4h822\\",\\"cols\\":[{\\"element_id\\":\\"far5824\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"fancy-heading\\",\\"element_id\\":\\"tfl2251\\",\\"mod_settings\\":{\\"heading\\":\\"Programs\\",\\"sub_heading\\":\\"build a stronger body today\\",\\"text_alignment\\":\\"themify-text-center\\",\\"heading_tag\\":\\"h3\\",\\"font_color_subheading\\":\\"#ffffff\\",\\"font_color_type_subheading\\":\\"font_color_subheading_solid\\",\\"font_color\\":\\"#ffffff\\",\\"font_color_type\\":\\"font_color_solid\\",\\"padding_bottom\\":\\"30\\",\\"background_position\\":\\"center-bottom\\",\\"background_repeat\\":\\"no-repeat\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bg-title-arrow-down.png\\",\\"breakpoint_tablet\\":{\\"padding_bottom\\":\\"30\\",\\"padding_bottom_unit\\":\\"px\\"}}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"gxvc671\\",\\"mod_settings\\":{\\"content_text\\":\\"<p>04<\\\\/p>\\",\\"font_size_h1_unit\\":\\"em\\",\\"font_size_h1\\":\\"10\\",\\"font_gradient_color_h1-gradient\\":\\"0% rgb(217, 88, 37)|100% rgb(173, 63, 16)\\",\\"font_color_type_h1\\":\\"font_gradient_color_h1_gradient\\",\\"font_color_type\\":\\"font_gradient_color_gradient\\",\\"custom_parallax_scroll_zindex\\":\\"1\\",\\"margin_bottom_unit\\":\\"%\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"margin_top_unit\\":\\"%\\",\\"margin_left_unit\\":\\"%\\",\\"margin_left\\":\\"-3\\",\\"margin_bottom\\":\\"-18\\",\\"breakpoint_mobile\\":{\\"margin_bottom_unit\\":\\"%\\",\\"margin_left_unit\\":\\"%\\",\\"margin_top_unit\\":\\"%\\",\\"margin_left\\":\\"-3\\",\\"margin_bottom\\":\\"-20\\"},\\"breakpoint_tablet\\":{\\"padding_bottom\\":\\"0\\",\\"margin_bottom_unit\\":\\"%\\",\\"margin_left_unit\\":\\"%\\",\\"margin_top_unit\\":\\"%\\",\\"margin_left\\":\\"-3\\",\\"margin_bottom\\":\\"-15\\",\\"padding_top\\":\\"30\\"},\\"line_height_h1_unit\\":\\"em\\",\\"line_height_h1\\":\\"1\\",\\"custom_parallax_scroll_speed\\":\\"1\\",\\"font_weight\\":\\"bold\\",\\"font_style_regular\\":\\"normal\\",\\"line_height_unit\\":\\"em\\",\\"line_height\\":\\"1\\",\\"font_size_unit\\":\\"em\\",\\"font_size\\":\\"10\\",\\"font_gradient_color-gradient\\":\\"0% rgb(217, 88, 37)|100% rgb(173, 63, 16)\\",\\"font_family_w\\":\\"800\\",\\"font_family\\":\\"Open Sans\\"}},{\\"element_id\\":\\"wtm3181\\",\\"cols\\":[{\\"element_id\\":\\"mfgr225\\",\\"grid_class\\":\\"col5-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"mjva226\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>CrossFit<\\\\/h3>\\\\n<p>Sunday <br \\\\/>08am - 10:30am<\\\\/p>\\\\n<h4>Benji Jr.<\\\\/h4>\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"40\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"40\\"}}}],\\"styling\\":{\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#ffffff\\",\\"cover_color_hover\\":\\"#cc4514_0.58\\",\\"cover_color_hover-type\\":\\"hover_color\\",\\"cover_color\\":\\"#14141b_0.56\\",\\"cover_color-type\\":\\"color\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"3\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"breakpoint_mobile\\":{\\"padding_right\\":\\"5\\",\\"padding_right_unit\\":\\"%\\",\\"border_right_style\\":\\"none\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom\\":\\"3\\",\\"padding_top\\":\\"3\\"},\\"breakpoint_tablet\\":{\\"border_bottom_width\\":\\"1\\",\\"border_bottom_color\\":\\"#cc4515\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"bottom\\"}}},{\\"element_id\\":\\"6d2s301\\",\\"grid_class\\":\\"col5-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"klzl301\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Zumba<\\\\/h3>\\\\n<p>Monday<br \\\\/>08am - 10:30am<\\\\/p>\\\\n<h4>Jessica  Kim<\\\\/h4>\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"40\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"40\\"}}}],\\"styling\\":{\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#ffffff\\",\\"cover_color_hover\\":\\"#cc4514_0.58\\",\\"cover_color_hover-type\\":\\"hover_color\\",\\"cover_color\\":\\"#14141b_0.56\\",\\"cover_color-type\\":\\"color\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"3\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"breakpoint_mobile\\":{\\"padding_right\\":\\"5\\",\\"padding_right_unit\\":\\"%\\",\\"border_right_style\\":\\"none\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom\\":\\"3\\",\\"padding_top\\":\\"3\\"},\\"breakpoint_tablet\\":{\\"border_bottom_width\\":\\"1\\",\\"border_bottom_color\\":\\"#cc4515\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"bottom\\"}}},{\\"element_id\\":\\"iy7z161\\",\\"grid_class\\":\\"col5-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"oliz161\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Muay Thai<\\\\/h3>\\\\n<p>Tuesday<br \\\\/>08am - 10:30am<\\\\/p>\\\\n<h4>Robert Moore<\\\\/h4>\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"40\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"40\\"}}}],\\"styling\\":{\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#ffffff\\",\\"cover_color_hover\\":\\"#cc4514_0.58\\",\\"cover_color_hover-type\\":\\"hover_color\\",\\"cover_color\\":\\"#14141b_0.56\\",\\"cover_color-type\\":\\"color\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"3\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"border_right_width\\":\\"1\\",\\"breakpoint_mobile\\":{\\"padding_right\\":\\"5\\",\\"padding_right_unit\\":\\"%\\",\\"border_right_style\\":\\"none\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom\\":\\"3\\",\\"padding_top\\":\\"3\\"},\\"breakpoint_tablet\\":{\\"border_bottom_width\\":\\"1\\",\\"border_bottom_color\\":\\"#cc4515\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"bottom\\"}}},{\\"element_id\\":\\"i81g272\\",\\"grid_class\\":\\"col5-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"n0wq273\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Yoga<\\\\/h3>\\\\n<p>Wednesday<br \\\\/>08am - 10:30am<\\\\/p>\\\\n<h4>Arunima Ali<\\\\/h4>\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"40\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"40\\"}}}],\\"styling\\":{\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#ffffff\\",\\"cover_color_hover\\":\\"#cc4514_0.58\\",\\"cover_color_hover-type\\":\\"hover_color\\",\\"cover_color\\":\\"#14141b_0.56\\",\\"cover_color-type\\":\\"color\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"3\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"breakpoint_mobile\\":{\\"padding_right\\":\\"5\\",\\"padding_right_unit\\":\\"%\\",\\"border_right_style\\":\\"none\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom\\":\\"3\\",\\"padding_top\\":\\"3\\"},\\"breakpoint_tablet\\":{\\"border_bottom_width\\":\\"1\\",\\"border_bottom_color\\":\\"#cc4515\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"bottom\\"}}},{\\"element_id\\":\\"y0qy666\\",\\"grid_class\\":\\"col5-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"ipx1498\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Pilates<\\\\/h3>\\\\n<p>Thursday<br \\\\/>08am - 10:30am<\\\\/p>\\\\n<h4>Michelina JuNG<\\\\/h4>\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"40\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"40\\"}}}],\\"styling\\":{\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#ffffff\\",\\"cover_color_hover\\":\\"#cc4514_0.58\\",\\"cover_color_hover-type\\":\\"hover_color\\",\\"cover_color\\":\\"#14141b_0.56\\",\\"cover_color-type\\":\\"color\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"3\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\"}}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet\\":\\"column3-1 tb_3col\\",\\"styling\\":{\\"border_bottom_width\\":\\"1\\",\\"border_bottom_color\\":\\"#cc4515\\",\\"border-type\\":\\"bottom\\",\\"zi\\":\\"2\\"}},{\\"element_id\\":\\"o9k1341\\",\\"cols\\":[{\\"element_id\\":\\"be37441\\",\\"grid_class\\":\\"col5-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"xndl441\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Body Pump<\\\\/h3>\\\\n<p>Friday<br \\\\/>08am - 10:30am<\\\\/p>\\\\n<h4>Arturo Dominic<\\\\/h4>\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"40\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"40\\"}}}],\\"styling\\":{\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#ffffff\\",\\"cover_color_hover\\":\\"#cc4514_0.58\\",\\"cover_color_hover-type\\":\\"hover_color\\",\\"cover_color\\":\\"#14141b_0.56\\",\\"cover_color-type\\":\\"color\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"3\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"breakpoint_mobile\\":{\\"padding_right\\":\\"5\\",\\"padding_right_unit\\":\\"%\\",\\"border_right_style\\":\\"none\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom\\":\\"3\\",\\"padding_top\\":\\"3\\"},\\"breakpoint_tablet\\":{\\"border_bottom_width\\":\\"1\\",\\"border_bottom_color\\":\\"#cc4515\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"bottom\\"}}},{\\"element_id\\":\\"r0zn767\\",\\"grid_class\\":\\"col5-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"bvcx769\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Booty Boot Camp<\\\\/h3>\\\\n<p>Saturday<br \\\\/>08am - 10:30am<\\\\/p>\\\\n<h4>Shania Dee<\\\\/h4>\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"40\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"40\\",\\"border_top_style\\":\\"none\\",\\"border-type\\":\\"top\\"}}}],\\"styling\\":{\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#ffffff\\",\\"cover_color_hover\\":\\"#cc4514_0.58\\",\\"cover_color_hover-type\\":\\"hover_color\\",\\"cover_color\\":\\"#14141b_0.56\\",\\"cover_color-type\\":\\"color\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"3\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"breakpoint_mobile\\":{\\"padding_right\\":\\"5\\",\\"padding_right_unit\\":\\"%\\",\\"border_right_style\\":\\"none\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom\\":\\"3\\",\\"padding_top\\":\\"3\\"},\\"breakpoint_tablet\\":{\\"border_bottom_width\\":\\"1\\",\\"border_bottom_color\\":\\"#cc4515\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"bottom\\"}}},{\\"element_id\\":\\"2p0p872\\",\\"grid_class\\":\\"col5-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"ls8w873\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Boxing 101<\\\\/h3>\\\\n<p>Sunday <br \\\\/>08am - 10:30am<\\\\/p>\\\\n<h4>Kenny Cheung<\\\\/h4>\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"40\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"40\\"}}}],\\"styling\\":{\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#ffffff\\",\\"cover_color_hover\\":\\"#cc4514_0.58\\",\\"cover_color_hover-type\\":\\"hover_color\\",\\"cover_color\\":\\"#14141b_0.56\\",\\"cover_color-type\\":\\"color\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"3\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"breakpoint_mobile\\":{\\"padding_right\\":\\"5\\",\\"padding_right_unit\\":\\"%\\",\\"border_right_style\\":\\"none\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom\\":\\"3\\",\\"padding_top\\":\\"3\\"},\\"breakpoint_tablet\\":{\\"border_bottom_width\\":\\"1\\",\\"border_bottom_color\\":\\"#cc4515\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"bottom\\"}}},{\\"element_id\\":\\"hynz383\\",\\"grid_class\\":\\"col5-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"r3iw384\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Barre Basics<\\\\/h3>\\\\n<p>Sunday<br \\\\/>08am - 10:30am<\\\\/p>\\\\n<h4>Olga Kane<\\\\/h4>\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"40\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"40\\"}}}],\\"styling\\":{\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#ffffff\\",\\"cover_color_hover\\":\\"#cc4514_0.58\\",\\"cover_color_hover-type\\":\\"hover_color\\",\\"cover_color\\":\\"#14141b_0.56\\",\\"cover_color-type\\":\\"color\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"3\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"breakpoint_mobile\\":{\\"padding_right\\":\\"5\\",\\"padding_right_unit\\":\\"%\\",\\"border_right_style\\":\\"none\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom\\":\\"3\\",\\"padding_top\\":\\"3\\"},\\"breakpoint_tablet\\":{\\"border_bottom_width\\":\\"1\\",\\"border_bottom_color\\":\\"#cc4515\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"bottom\\"}}},{\\"element_id\\":\\"xqf4557\\",\\"grid_class\\":\\"col5-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"z0xp558\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Spinning<\\\\/h3>\\\\n<p>Sunday <br \\\\/>08am - 10:30am<\\\\/p>\\\\n<h4>Michael Okue<\\\\/h4>\\",\\"breakpoint_mobile\\":{\\"padding_bottom\\":\\"40\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"40\\"}}}],\\"styling\\":{\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"3\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#ffffff\\",\\"cover_color_hover\\":\\"#cc4514_0.58\\",\\"cover_color_hover-type\\":\\"hover_color\\",\\"cover_color\\":\\"#14141b_0.56\\",\\"cover_color-type\\":\\"color\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"3\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"breakpoint_mobile\\":{\\"padding_right\\":\\"5\\",\\"padding_right_unit\\":\\"%\\",\\"border_right_style\\":\\"none\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_bottom\\":\\"3\\",\\"padding_top\\":\\"3\\"},\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"right\\",\\"breakpoint_tablet\\":{\\"border_bottom_width\\":\\"1\\",\\"border_bottom_color\\":\\"#cc4515\\",\\"border_right_width\\":\\"1\\",\\"border_right_color\\":\\"#cc4515\\",\\"border-type\\":\\"bottom\\"}}}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet\\":\\"column3-1 tb_3col\\",\\"styling\\":{\\"breakpoint_mobile\\":{\\"border-type\\":\\"left\\",\\"border_left_style\\":\\"none\\"}}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"74nq478\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Book Now\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/contact\\\\/\\",\\"link_options\\":\\"regular\\",\\"icon\\":\\"ti-arrow-right\\",\\"icon_alignment\\":\\"right\\",\\"button_color_bg\\":\\"tb_default_color\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_shape\\":\\"squared\\",\\"buttons_size\\":\\"large\\",\\"alignment\\":\\"center\\",\\"margin_bottom_unit\\":\\"%\\",\\"margin_bottom\\":\\"5\\",\\"margin_top_unit\\":\\"%\\",\\"margin_top\\":\\"5\\",\\"link_color\\":\\"#ffffff\\"}}]}],\\"gutter\\":\\"gutter-none\\",\\"styling\\":{\\"row_width\\":\\"fullwidth-content\\",\\"background_position\\":\\"center-top\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bg-join-programs.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"padding_bottom\\":\\"8\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"7\\",\\"row_anchor\\":\\"programs\\"}},{\\"element_id\\":\\"8rrl778\\",\\"cols\\":[{\\"element_id\\":\\"lc6e779\\",\\"grid_class\\":\\"col4-2\\"},{\\"element_id\\":\\"ufig691\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"fancy-heading\\",\\"element_id\\":\\"nrfv691\\",\\"mod_settings\\":{\\"heading\\":\\"Our Locations\\",\\"sub_heading\\":\\"map & addresses\\",\\"text_alignment\\":\\"themify-text-center\\",\\"heading_tag\\":\\"h3\\",\\"padding_bottom\\":\\"30\\",\\"background_position\\":\\"center-bottom\\",\\"background_repeat\\":\\"no-repeat\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/bg-title-arrow-down.png\\",\\"margin_bottom\\":\\"30\\"}},{\\"mod_name\\":\\"icon\\",\\"element_id\\":\\"a3yg691\\",\\"mod_settings\\":{\\"content_icon\\":[{\\"icon\\":\\"ti-location-pin\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"84 Dundas St W, Toronto, ON, Canada\\",\\"link_options\\":\\"regular\\"},{\\"icon\\":\\"ti-location-pin\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"343 Bathurst St, Toronto, ON, Canada\\",\\"link_options\\":\\"regular\\"},{\\"icon\\":\\"ti-location-pin\\",\\"icon_color_bg\\":\\"transparent\\",\\"label\\":\\"101 Carr St, Toronto, ON, Canada\\",\\"link_options\\":\\"regular\\"}],\\"icon_position\\":\\"icon_position_left\\",\\"icon_arrangement\\":\\"icon_vertical\\",\\"icon_style\\":\\"circle\\",\\"icon_size\\":\\"large\\"}}],\\"styling\\":{\\"padding_left_unit\\":\\"%\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"8\\",\\"padding_right_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"8\\",\\"background_color\\":\\"#ffffff\\",\\"background_position\\":\\"center-center\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"padding_left\\":\\"5\\",\\"padding_right\\":\\"5\\",\\"breakpoint_tablet\\":{\\"text_align\\":\\"center\\"},\\"b_sh_inset\\":false,\\"b_sh_color\\":\\"#000000_0.10\\",\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_blur\\":\\"18\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_vOffset\\":\\"4\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"b_sh_hOffset\\":\\"0\\",\\"checkbox_padding_apply_all\\":false,\\"padding_opp_left\\":false,\\"padding_opp_top\\":false}}],\\"styling\\":{\\"row_width\\":\\"fullwidth\\",\\"margin-top\\":\\"-1\\",\\"margin-top_unit\\":\\"px\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"12\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"12\\",\\"background_position\\":\\"center-top\\",\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-trainers\\\\/files\\\\/2019\\\\/07\\\\/map-big.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"breakpoint_mobile\\":{\\"padding_top\\":\\"15\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom\\":\\"15\\",\\"padding_bottom_unit\\":\\"%\\"}}}]',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 159,
  'post_date' => '2019-07-24 01:52:57',
  'post_date_gmt' => '2019-07-24 01:52:57',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '159',
  'post_modified' => '2019-07-31 02:32:08',
  'post_modified_gmt' => '2019-07-31 02:32:08',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=159',
  'menu_order' => 1,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '7',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'footer-widget-1',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 452,
  'post_date' => '2019-07-31 02:31:50',
  'post_date_gmt' => '2019-07-31 02:31:50',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '452',
  'post_modified' => '2019-07-31 02:32:08',
  'post_modified_gmt' => '2019-07-31 02:32:08',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=452',
  'menu_order' => 2,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '173',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'footer-widget-1',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 450,
  'post_date' => '2019-07-31 02:31:50',
  'post_date_gmt' => '2019-07-31 02:31:50',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '450',
  'post_modified' => '2019-07-31 02:32:08',
  'post_modified_gmt' => '2019-07-31 02:32:08',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=450',
  'menu_order' => 3,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '179',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'footer-widget-1',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 168,
  'post_date' => '2019-07-24 02:25:59',
  'post_date_gmt' => '2019-07-24 02:25:59',
  'post_content' => '',
  'post_title' => 'Programs',
  'post_excerpt' => '',
  'post_name' => 'programs',
  'post_modified' => '2019-07-31 02:32:08',
  'post_modified_gmt' => '2019-07-31 02:32:08',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=168',
  'menu_order' => 4,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'custom',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '168',
    '_menu_item_object' => 'custom',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
    '_menu_item_url' => 'https://themify.me/demo/themes/ultra-trainers/#programs',
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'footer-widget-1',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 451,
  'post_date' => '2019-07-31 02:31:50',
  'post_date_gmt' => '2019-07-31 02:31:50',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '451',
  'post_modified' => '2019-07-31 02:32:08',
  'post_modified_gmt' => '2019-07-31 02:32:08',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=451',
  'menu_order' => 5,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '176',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'footer-widget-1',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 453,
  'post_date' => '2019-07-31 02:33:04',
  'post_date_gmt' => '2019-07-31 02:33:04',
  'post_content' => '',
  'post_title' => 'Programs',
  'post_excerpt' => '',
  'post_name' => 'programs-2',
  'post_modified' => '2019-07-31 02:37:20',
  'post_modified_gmt' => '2019-07-31 02:37:20',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=453',
  'menu_order' => 1,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'custom',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '453',
    '_menu_item_object' => 'custom',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
    '_menu_item_url' => 'https://themify.me/demo/themes/ultra-trainers/#programs',
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'footer-widget-2',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 456,
  'post_date' => '2019-07-31 02:36:41',
  'post_date_gmt' => '2019-07-31 02:36:41',
  'post_content' => '',
  'post_title' => 'Events',
  'post_excerpt' => '',
  'post_name' => 'events',
  'post_modified' => '2019-07-31 02:37:20',
  'post_modified_gmt' => '2019-07-31 02:37:20',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=456',
  'menu_order' => 2,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'custom',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '456',
    '_menu_item_object' => 'custom',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
    '_menu_item_url' => 'https://themify.me/demo/themes/ultra-trainers/activities/#events',
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'footer-widget-2',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 458,
  'post_date' => '2019-07-31 02:36:41',
  'post_date_gmt' => '2019-07-31 02:36:41',
  'post_content' => '',
  'post_title' => 'Apparels',
  'post_excerpt' => '',
  'post_name' => 'apparels',
  'post_modified' => '2019-07-31 02:37:20',
  'post_modified_gmt' => '2019-07-31 02:37:20',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=458',
  'menu_order' => 3,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'custom',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '458',
    '_menu_item_object' => 'custom',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
    '_menu_item_url' => 'https://themify.me/demo/themes/ultra-trainers/about/#apparel',
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'footer-widget-2',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 459,
  'post_date' => '2019-07-31 02:36:41',
  'post_date_gmt' => '2019-07-31 02:36:41',
  'post_content' => '',
  'post_title' => 'Trainers',
  'post_excerpt' => '',
  'post_name' => 'trainers',
  'post_modified' => '2019-07-31 02:37:20',
  'post_modified_gmt' => '2019-07-31 02:37:20',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=459',
  'menu_order' => 4,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'custom',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '459',
    '_menu_item_object' => 'custom',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
    '_menu_item_url' => 'https://themify.me/demo/themes/ultra-trainers/about/#trainers',
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'footer-widget-2',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 457,
  'post_date' => '2019-07-31 02:36:41',
  'post_date_gmt' => '2019-07-31 02:36:41',
  'post_content' => '',
  'post_title' => 'Gallery',
  'post_excerpt' => '',
  'post_name' => 'gallery',
  'post_modified' => '2019-07-31 02:37:20',
  'post_modified_gmt' => '2019-07-31 02:37:20',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=457',
  'menu_order' => 5,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'custom',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '457',
    '_menu_item_object' => 'custom',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
    '_menu_item_url' => 'https://themify.me/demo/themes/ultra-trainers/activities/#gallery',
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'footer-widget-2',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 158,
  'post_date' => '2019-07-24 01:50:57',
  'post_date_gmt' => '2019-07-24 01:50:57',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '158',
  'post_modified' => '2019-07-29 08:42:58',
  'post_modified_gmt' => '2019-07-29 08:42:58',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=158',
  'menu_order' => 1,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '7',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'main-navigation',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 195,
  'post_date' => '2019-07-29 08:42:58',
  'post_date_gmt' => '2019-07-29 08:42:58',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '195',
  'post_modified' => '2019-07-29 08:42:58',
  'post_modified_gmt' => '2019-07-29 08:42:58',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=195',
  'menu_order' => 2,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '173',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'main-navigation',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 193,
  'post_date' => '2019-07-29 08:42:58',
  'post_date_gmt' => '2019-07-29 08:42:58',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '193',
  'post_modified' => '2019-07-29 08:42:58',
  'post_modified_gmt' => '2019-07-29 08:42:58',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=193',
  'menu_order' => 3,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '179',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'main-navigation',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 194,
  'post_date' => '2019-07-29 08:42:58',
  'post_date_gmt' => '2019-07-29 08:42:58',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '194',
  'post_modified' => '2019-07-29 08:42:58',
  'post_modified_gmt' => '2019-07-29 08:42:58',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-trainers/?p=194',
  'menu_order' => 4,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '176',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'main-navigation',
  ),
);
themify_process_post_import( $post );



function themify_import_get_term_id_from_slug( $slug ) {
	$menu = get_term_by( "slug", $slug, "nav_menu" );
	return is_wp_error( $menu ) ? 0 : (int) $menu->term_id;
}

	$widgets = get_option( "widget_nav_menu" );
$widgets[1002] = array (
  'title' => 'Sitemap',
  'nav_menu' => themify_import_get_term_id_from_slug( "footer-widget-1" ),
);
update_option( "widget_nav_menu", $widgets );

$widgets = get_option( "widget_nav_menu" );
$widgets[1003] = array (
  'title' => 'Links',
  'nav_menu' => themify_import_get_term_id_from_slug( "footer-widget-2" ),
);
update_option( "widget_nav_menu", $widgets );

$widgets = get_option( "widget_text" );
$widgets[1004] = array (
  'title' => 'Contact',
  'text' => '<i class="ti-location-pin"></i>84 Dundas St W, Toronto, ON, Canada

<i class="ti-mobile"></i> +1-416-777-0777

<i class="ti-email"></i> Info@UltraTrainers.com',
  'filter' => true,
  'visual' => true,
);
update_option( "widget_text", $widgets );



$sidebars_widgets = array (
  'footer-widget-1' => 
  array (
    0 => 'nav_menu-1002',
  ),
  'footer-widget-2' => 
  array (
    0 => 'nav_menu-1003',
  ),
  'footer-widget-3' => 
  array (
    0 => 'text-1004',
  ),
); 
update_option( "sidebars_widgets", $sidebars_widgets );

$homepage = get_posts( array( 'name' => 'home', 'post_type' => 'page' ) );
			if( is_array( $homepage ) && ! empty( $homepage ) ) {
				update_option( 'show_on_front', 'page' );
				update_option( 'page_on_front', $homepage[0]->ID );
			}
			$themify_data = array (
  'setting-webfonts_list' => 'recommended',
  'setting-default_layout' => 'sidebar1',
  'setting-default_post_layout' => 'list-post',
  'setting-post_filter' => 'no',
  'setting-disable_masonry' => 'yes',
  'setting-post_gutter' => 'gutter',
  'setting-default_layout_display' => 'content',
  'setting-default_more_text' => 'More',
  'setting-index_orderby' => 'date',
  'setting-index_order' => 'DESC',
  'setting-default_media_position' => 'above',
  'setting-image_post_feature_size' => 'blank',
  'setting-default_page_post_layout' => 'sidebar1',
  'setting-default_page_post_layout_type' => 'classic',
  'setting-default_page_single_media_position' => 'above',
  'setting-image_post_single_feature_size' => 'blank',
  'setting-search-result_layout_display' => 'content',
  'setting-search-result_media_position' => 'above',
  'setting-default_page_layout' => 'sidebar1',
  'setting-custom_post_tglobal_style_single' => 'sidebar1',
  'setting-default_portfolio_index_layout' => 'sidebar-none',
  'setting-default_portfolio_index_post_layout' => 'grid3',
  'setting-portfolio_post_filter' => 'yes',
  'setting-portfolio_disable_masonry' => 'yes',
  'setting-portfolio_gutter' => 'gutter',
  'setting-default_portfolio_index_display' => 'none',
  'setting-default_portfolio_index_title' => ' ',
  'setting-default_portfolio_index_unlink_post_title' => ' ',
  'setting-default_portfolio_index_hide_post_date' => ' ',
  'setting-default_portfolio_index_post_meta_category' => 'yes',
  'setting-default_portfolio_index_unlink_post_image' => 'yes',
  'setting-default_portfolio_single_layout' => 'sidebar-none',
  'setting-default_portfolio_single_portfolio_layout_type' => 'fullwidth',
  'setting-default_portfolio_single_title' => ' ',
  'setting-default_portfolio_single_unlink_post_title' => ' ',
  'setting-default_portfolio_single_hide_post_date' => ' ',
  'setting-default_portfolio_single_post_meta_category' => ' ',
  'setting-default_portfolio_single_unlink_post_image' => 'yes',
  'themify_portfolio_slug' => 'project',
  'themify_portfolio_category_slug' => 'portfolio-category',
  'setting-customizer_responsive_design_tablet_landscape' => '1280',
  'setting-customizer_responsive_design_tablet' => '768',
  'setting-customizer_responsive_design_mobile' => '680',
  'setting-mobile_menu_trigger_point' => '900',
  'setting-gallery_lightbox' => 'lightbox',
  'setting-header_design' => 'header-horizontal',
  'setting-exclude_site_tagline' => 'on',
  'setting-exclude_search_form' => 'on',
  'setting_search_form' => 'live_search',
  'setting-exclude_rss' => 'on',
  'setting-header_widgets' => 'headerwidget-3col',
  'setting-footer_design' => 'footer-left-col',
  'setting-use_float_back' => 'on',
  'setting-footer_widgets' => 'footerwidget-3col',
  'setting-mega_menu_posts' => '5',
  'setting-mega_menu_image_width' => '180',
  'setting-mega_menu_image_height' => '120',
  'setting-mega_menu_post_count' => 'off',
  'setting-imagefilter_applyto' => 'featuredonly',
  'setting-color_animation_speed' => '5',
  'setting-relationship_taxonomy' => 'category',
  'setting-relationship_taxonomy_entries' => '3',
  'setting-relationship_taxonomy_display_content' => 'none',
  'setting-single_slider_autoplay' => 'off',
  'setting-single_slider_speed' => 'normal',
  'setting-single_slider_effect' => 'slide',
  'setting-single_slider_height' => 'auto',
  'setting-more_posts' => 'infinite',
  'setting-entries_nav' => 'numbered',
  'setting-footer_text_left' => 'Build your body and your website with the Ultra Trainers skin. The Ultra theme is the ultimate all-in-one WordPress theme and the base theme for the demo site you\'re viewing. Discover the entire collection of Ultra skins today!',
  'setting-img_php_base_size' => 'large',
  'setting-global_feature_size' => 'blank',
  'setting-link_icon_type' => 'font-icon',
  'setting-link_type_themify-link-0' => 'image-icon',
  'setting-link_title_themify-link-0' => 'Twitter',
  'setting-link_img_themify-link-0' => 'https://themify.me/demo/themes/ultra-trainers/wp-content/themes/themify-ultra/themify/img/social/twitter.png',
  'setting-link_type_themify-link-1' => 'image-icon',
  'setting-link_title_themify-link-1' => 'Facebook',
  'setting-link_img_themify-link-1' => 'https://themify.me/demo/themes/ultra-trainers/wp-content/themes/themify-ultra/themify/img/social/facebook.png',
  'setting-link_type_themify-link-2' => 'image-icon',
  'setting-link_title_themify-link-2' => 'YouTube',
  'setting-link_img_themify-link-2' => 'https://themify.me/demo/themes/ultra-trainers/wp-content/themes/themify-ultra/themify/img/social/youtube.png',
  'setting-link_type_themify-link-3' => 'image-icon',
  'setting-link_title_themify-link-3' => 'Pinterest',
  'setting-link_img_themify-link-3' => 'https://themify.me/demo/themes/ultra-trainers/wp-content/themes/themify-ultra/themify/img/social/pinterest.png',
  'setting-link_type_themify-link-4' => 'font-icon',
  'setting-link_title_themify-link-4' => 'Twitter',
  'setting-link_ficon_themify-link-4' => 'fa-twitter',
  'setting-link_type_themify-link-5' => 'font-icon',
  'setting-link_title_themify-link-5' => 'Facebook',
  'setting-link_ficon_themify-link-5' => 'fa-facebook',
  'setting-link_type_themify-link-6' => 'font-icon',
  'setting-link_title_themify-link-6' => 'YouTube',
  'setting-link_ficon_themify-link-6' => 'fa-youtube',
  'setting-link_type_themify-link-7' => 'font-icon',
  'setting-link_title_themify-link-7' => 'Pinterest',
  'setting-link_ficon_themify-link-7' => 'fa-pinterest',
  'setting-link_field_ids' => '{"themify-link-0":"themify-link-0","themify-link-1":"themify-link-1","themify-link-2":"themify-link-2","themify-link-3":"themify-link-3","themify-link-4":"themify-link-4","themify-link-5":"themify-link-5","themify-link-6":"themify-link-6","themify-link-7":"themify-link-7"}',
  'setting-link_field_hash' => '8',
  'setting-page_builder_is_active' => 'enable',
  'setting-page_builder_animation_appearance' => 'none',
  'setting-page_builder_animation_parallax_bg' => 'none',
  'setting-page_builder_animation_scroll_effect' => 'none',
  'setting-page_builder_animation_sticky_scroll' => 'none',
  'setting-page_builder_expiry' => '2',
  'skin' => 'trainers',
  'import_images' => 'on',
);
themify_set_data( $themify_data );
$theme = get_option( 'stylesheet' );
$theme_mods = array (
  0 => false,
  'custom_css_post_id' => -1,
);
update_option( "theme_mods_{$theme}", $theme_mods );
$menu_locations = array();
$menu = get_terms( "nav_menu", array( "slug" => "main-navigation" ) );
if( is_array( $menu ) && ! empty( $menu ) ) $menu_locations["main-nav"] = $menu[0]->term_id;
set_theme_mod( "nav_menu_locations", $menu_locations );



}
themify_do_demo_import();