<?php

namespace ToStartWork\Http\Controllers\Api;

use iio\libmergepdf\Merger;
use Illuminate\Http\Request;
use ToStartWork\Http\Controllers\Controller;
use ToStartWork\Mail\QualificationPdfGenerated;
use Nitmedia\Wkhtml2pdf\Facades\Wkhtml2pdf as PDF;
use Illuminate\Support\Facades\Log;
use ToStartWork\Helpers\Misc;

class QualificationPdfController extends Controller
{
    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function downloadPdf(Request $request)
    {
        try {
            Log::info('Show qualification request data: ', $request->all());

            $user = \Auth::user();

            $qualifications = $this->getQualifications($request, $user);

            $includes = [
                'include_personal_details' => $request->get('include_personal_details'),
                'include_licenses' => $request->get('include_licenses'),
                'include_medical_information' => $request->get('include_medical_information'),
                'include_emergency_contacts' => $request->get('include_emergency_contacts'),
                'include_employment_details' => $request->get('include_employment_details')
            ];

            $userData = [
                'first_name' => $user->first_name,
                'last_name' => $user->last_name,
                'email' => $user->email,
                'contact_phone_no' => $user->contact_phone_no,
                'contact_email' => $user->contact_email,
                'address' => $user->address,
                'user_image_url' => $user->user_image_url,
                'name' => $user->name,
            ];

            // get personal details
            if ($includes['include_personal_details']) {
                $userData['middle_name'] = $user->middle_name;
                $userData['nickname'] = $user->nickname;
                $userData['gender'] = ucfirst($user->gender);
                $userData['dob'] = date('d/m/Y', strtotime($user->dob));
            }

            // get licenses
            if ($includes['include_licenses']) {
                $userData['citb_no'] = $user->citb_no;
                $userData['citb_expiry_date'] = $user->citb_expiry_date;
                $userData['drivers_license_class'] = $user->drivers_license_class;
                $userData['drivers_license_no'] = $user->drivers_license_no;
                $userData['drivers_license_class'] = $user->drivers_license_class;
                $userData['drivers_license_name'] = $user->drivers_license_name;
                $userData['drivers_license_expiry_date'] = $user->drivers_license_expiry_date;
            }

            // get medical information
            if ($includes['include_medical_information']) {
                $userData['has_tetanus_injection'] = $user->has_tetanus_injection;
                $userData['tetanus_injection_details'] = $user->tetanus_injection_details;
                $userData['has_epilepsy'] = $user->has_epilepsy;
                $userData['epilepsy_details'] = $user->epilepsy_details;
                $userData['has_diabetes'] = $user->has_diabetes;
                $userData['diabetes_details'] = $user->diabetes_details;
                $userData['has_heart_condition'] = $user->has_heart_condition;
                $userData['heart_condition_details'] = $user->heart_condition_details;
                $userData['has_medical_condition'] = $user->has_medical_condition;
                $userData['medical_condition_details'] = $user->medical_condition_details;
                $userData['has_taken_medication'] = $user->has_taken_medication;
                $userData['taken_medication_details'] = $user->taken_medication_details;
                $userData['has_allergies'] = $user->has_allergies;
                $userData['allergies_details'] = $user->allergies_details;
            }

            // get emergency contacts
            if ($includes['include_emergency_contacts']) {
                $userData['emergency_contacts'] = $user->contacts;
            }

            // get employment details
            if ($includes['include_employment_details']) {
                $userData['trades'] = $user->trades;
                $userData['employment_details'] = $user->employment;
            }

            PDF::setOrientation('portrait');
            PDF::setOutputMode('S');

            $cover = $this->getPdfCover($userData, $includes);
            $pdf = $this->getPdfMain($userData, $includes, $qualifications);

            // need to merge cover and main pdf together
            $pdfMerger = new Merger();
            $pdfMerger->addRaw($cover);
            $pdfMerger->addRaw($pdf);
            $pdf = $pdfMerger->merge();

            // return response($pdf, 200)->header('Content-Type', 'application/pdf')->header('Content-Disposition', 'attachment; filename="induction.pdf"');

            // Then we send email to receiver email and attach the pdf
            $receiver = $request->get('receiver');

            if (array_get($receiver, 'email')) {
                \Mail::to($receiver['email'])->send(new QualificationPdfGenerated($pdf, $receiver, $user));
                $msg = 'Successfully sent email to ' . $receiver['email'];

                Log::info($msg);
                return response()->json(['message' => $msg], 200);
            }

            return response()->json(['message' => 'No email found'], 404);
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return response()->json(['message' => $e->getMessage()], 200);
        }
    }

    private function getQualifications(Request $request, $user)
    {
        $qualificationIds = array_pluck($request->input('qualifications'), 'id');

        $qualifications = $user->qualifications()
            ->whereIn('id', $qualificationIds)
            ->withPivot(['front_image', 'back_image', 'ref_no', 'issue_date', 'expiry_date'])->get()->toArray();

        foreach ($qualifications as $key => $qualification) {
            foreach (['front', 'back'] as $name) {
                if (array_get($qualification, "pivot.{$name}_image")) {
                    $qualification['pivot']["{$name}_image"] = Misc::encodeImageToBase64($qualification['pivot']["{$name}_image"]);
                }
            }

            $qualifications[$key] = $qualification;
        }

        return $qualifications;
    }

    private function getPdfCover($userData, $includes)
    {
        PDF::setOptions("--javascript-delay 1000 --no-stop-slow-scripts --debug-javascript --enable-smart-shrinking -T 0 -B 0 -L 0 -R 0 ");
        return PDF::html('pdf.cover', compact('userData', 'includes'));
    }

    private function getPdfMain($userData, $includes, $qualifications)
    {
        $header = $this->getPdfPartial('header');
        $footer = $this->getPdfPartial('footer');

        PDF::setOptions("--javascript-delay 1000 --no-stop-slow-scripts --debug-javascript --enable-smart-shrinking --header-spacing 5 --footer-spacing 0 -B 35 -L 0 -R 0 -T 35 --header-html '{$header}' --footer-html '{$footer}'");

        return PDF::html('pdf.main', compact('userData', 'includes', 'qualifications'));
    }

    private function getPdfPartial($name)
    {
        $path = public_path("qualification-{$name}.html");

        file_put_contents($path, view("pdf.{$name}")->render());

        return $path;
    }
}
