<?php

defined( 'ABSPATH' ) or die;

$GLOBALS['processed_terms'] = array();
$GLOBALS['processed_posts'] = array();

require_once ABSPATH . 'wp-admin/includes/post.php';
require_once ABSPATH . 'wp-admin/includes/taxonomy.php';
require_once ABSPATH . 'wp-admin/includes/image.php';

/**
 * Add an Import Action, this data is stored for processing after import is done.
 *
 * Each action is sent as an Ajax request and is handled by themify-ajax.php file
 */ 
function themify_add_import_action( $action = '', $data = array() ) {
	global $import_actions;

	if ( ! isset( $import_actions[ $action ] ) ) {
		$import_actions[ $action ] = array();
	}

	$import_actions[ $action ][] = $data;
}

function themify_import_post( $post ) {
	global $processed_posts, $processed_terms;

	if ( ! post_type_exists( $post['post_type'] ) ) {
		return;
	}

	/* Menu items don't have reliable post_title, skip the post_exists check */
	if( $post['post_type'] !== 'nav_menu_item' ) {
		$post_exists = post_exists( $post['post_title'], '', $post['post_date'] );
		if ( $post_exists && get_post_type( $post_exists ) == $post['post_type'] ) {
			$processed_posts[ intval( $post['ID'] ) ] = intval( $post_exists );
			return;
		}
	}

	if( $post['post_type'] == 'nav_menu_item' ) {
		if( ! isset( $post['tax_input']['nav_menu'] ) || ! term_exists( $post['tax_input']['nav_menu'], 'nav_menu' ) ) {
			return;
		}
		$_menu_item_type = $post['meta_input']['_menu_item_type'];
		$_menu_item_object_id = $post['meta_input']['_menu_item_object_id'];

		if ( 'taxonomy' == $_menu_item_type && isset( $processed_terms[ intval( $_menu_item_object_id ) ] ) ) {
			$post['meta_input']['_menu_item_object_id'] = $processed_terms[ intval( $_menu_item_object_id ) ];
		} else if ( 'post_type' == $_menu_item_type && isset( $processed_posts[ intval( $_menu_item_object_id ) ] ) ) {
			$post['meta_input']['_menu_item_object_id'] = $processed_posts[ intval( $_menu_item_object_id ) ];
		} else if ( 'custom' != $_menu_item_type ) {
			// associated object is missing or not imported yet, we'll retry later
			// $missing_menu_items[] = $item;
			return;
		}
	}

	$post_parent = ( $post['post_type'] == 'nav_menu_item' ) ? $post['meta_input']['_menu_item_menu_item_parent'] : (int) $post['post_parent'];
	$post['post_parent'] = 0;
	if ( $post_parent ) {
		// if we already know the parent, map it to the new local ID
		if ( isset( $processed_posts[ $post_parent ] ) ) {
			if( $post['post_type'] == 'nav_menu_item' ) {
				$post['meta_input']['_menu_item_menu_item_parent'] = $processed_posts[ $post_parent ];
			} else {
				$post['post_parent'] = $processed_posts[ $post_parent ];
			}
		}
	}

	/**
	 * for hierarchical taxonomies, IDs must be used so wp_set_post_terms can function properly
	 * convert term slugs to IDs for hierarchical taxonomies
	 */
	if( ! empty( $post['tax_input'] ) ) {
		foreach( $post['tax_input'] as $tax => $terms ) {
			if( is_taxonomy_hierarchical( $tax ) ) {
				$terms = explode( ', ', $terms );
				$post['tax_input'][ $tax ] = array_map( 'themify_get_term_id_by_slug', $terms, array_fill( 0, count( $terms ), $tax ) );
			}
		}
	}

	$post['post_author'] = (int) get_current_user_id();
	$post['post_status'] = 'publish';

	$old_id = $post['ID'];

	unset( $post['ID'] );
	$post_id = wp_insert_post( $post, true );
	if( is_wp_error( $post_id ) ) {
		return false;
	} else {
		$processed_posts[ $old_id ] = $post_id;

		return $post_id;
	}
}

function themify_get_placeholder_image() {
	static $placeholder_image = null;

	if ( $placeholder_image == null ) {
		if ( ! function_exists( 'WP_Filesystem' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}
		WP_Filesystem();
		global $wp_filesystem;

		$post = array();

		$upload = wp_upload_bits( 'themify-placeholder.jpg', null, $wp_filesystem->get_contents( THEMIFY_DIR . '/img/image-placeholder.jpg' ) );

		if ( $info = wp_check_filetype( $upload['file'] ) )
			$post = array(
				'post_mime_type' => $info['type']
			);
		else
			return new WP_Error( 'attachment_processing_error', __( 'Invalid file type', 'themify' ) );

		$post['guid'] = $upload['url'];
		$placeholder_image = wp_insert_attachment( $post, $upload['file'] );
		require_once( ABSPATH . 'wp-admin/includes/image.php' );
		wp_update_attachment_metadata( $placeholder_image, wp_generate_attachment_metadata( $placeholder_image, $upload['file'] ) );
	}

	return $placeholder_image;
}

function themify_import_term( $term ) {
	global $processed_terms;

	if( $term_id = term_exists( $term['slug'], $term['taxonomy'] ) ) {
		if ( is_array( $term_id ) ) $term_id = $term_id['term_id'];
		if ( isset( $term['term_id'] ) )
			$processed_terms[ intval( $term['term_id'] ) ] = (int) $term_id;
		return (int) $term_id;
	}

	if ( empty( $term['parent'] ) || ! isset( $processed_terms[ intval( $term['parent'] ) ] ) ) {
		$parent = 0;
	} else {
		$parent = term_exists( $processed_terms[ intval( $term['parent'] ) ], $term['taxonomy'] );
		if ( is_array( $parent ) ) $parent = $parent['term_id'];
	}

	$id = wp_insert_term( $term['name'], $term['taxonomy'], array(
		'parent' => $parent,
		'slug' => $term['slug'],
		'description' => $term['description'],
	) );
	if ( ! is_wp_error( $id ) ) {
		if ( isset( $term['term_id'] ) ) {
			// success!
			$processed_terms[ intval($term['term_id']) ] = $id['term_id'];
			if ( isset( $term['thumbnail'] ) ) {
				themify_add_import_action( 'term_thumb', array(
					'id' => $id['term_id'],
					'thumb' => $term['thumbnail'],
				) );
			}
			return $term['term_id'];
		}
	}

	return false;
}

function themify_get_term_id_by_slug( $slug, $tax ) {
	$term = get_term_by( 'slug', $slug, $tax );
	if( $term ) {
		return $term->term_id;
	}

	return false;
}

function themify_undo_import_term( $term ) {
	$term_id = term_exists( $term['slug'], $term['taxonomy'] );
	if ( $term_id ) {
		if ( is_array( $term_id ) ) $term_id = $term_id['term_id'];

		if ( $term_thumbnail = get_term_meta( $term['term_id'], 'thumbnail_id', true ) ) {
			wp_delete_attachment( $term_thumbnail, true );
		}

		if ( isset( $term_id ) ) {
			wp_delete_term( $term_id, $term['taxonomy'] );
		}
	}
}

/**
 * Determine if a post exists based on title, content, and date
 *
 * @global wpdb $wpdb WordPress database abstraction object.
 *
 * @param array $args array of database parameters to check
 * @return int Post ID if post exists, 0 otherwise.
 */
function themify_post_exists( $args = array() ) {
	global $wpdb;

	$query = "SELECT ID FROM $wpdb->posts WHERE 1=1";
	$db_args = array();

	foreach ( $args as $key => $value ) {
		$value = wp_unslash( sanitize_post_field( $key, $value, 0, 'db' ) );
		if( ! empty( $value ) ) {
			$query .= ' AND ' . $key . ' = %s';
			$db_args[] = $value;
		}
	}

	if ( !empty ( $args ) )
		return (int) $wpdb->get_var( $wpdb->prepare($query, $args) );

	return 0;
}

function themify_undo_import_post( $post ) {
	if( $post['post_type'] == 'nav_menu_item' ) {
		$post_exists = themify_post_exists( array(
			'post_name' => $post['post_name'],
			'post_modified' => $post['post_date'],
			'post_type' => 'nav_menu_item',
		) );
	} else {
		$post_exists = post_exists( $post['post_title'], '', $post['post_date'] );
	}
	if( $post_exists && get_post_type( $post_exists ) == $post['post_type'] ) {
		/**
		 * check if the post has been modified, if so leave it be
		 *
		 * NOTE: posts are imported using wp_insert_post() which modifies post_modified field
		 * to be the same as post_date, hence to check if the post has been modified,
		 * the post_modified field is compared against post_date in the original post.
		 */
		if( $post['post_date'] == get_post_field( 'post_modified', $post_exists ) ) {
			// find and remove all post attachments
			$attachments = get_posts( array(
				'post_type' => 'attachment',
				'posts_per_page' => -1,
				'post_parent' => $post_exists,
			) );
			if ( $attachments ) {
				foreach ( $attachments as $attachment ) {
					wp_delete_attachment( $attachment->ID, true );
				}
			}
			wp_delete_post( $post_exists, true ); // true: bypass trash
		}
	}
}

function themify_process_post_import( $post ) {
	if( ERASEDEMO ) {
		themify_undo_import_post( $post );
	} else {
		if ( $id = themify_import_post( $post ) ) {
			if ( defined( 'IMPORT_IMAGES' ) && ! IMPORT_IMAGES ) {
				/* if importing images is disabled and post is supposed to have a thumbnail, create a placeholder image instead */
				if ( isset( $post['thumb'] ) ) { // the post is supposed to have featured image
					$placeholder = themify_get_placeholder_image();
					if ( ! empty( $placeholder ) && ! is_wp_error( $placeholder ) ) {
						set_post_thumbnail( $id, $placeholder );
					}
				}
			} else {
				if ( isset( $post["thumb"] ) ) {
					themify_add_import_action( 'post_thumb', array(
						'id' => $id,
						'thumb' => $post["thumb"],
					) );
				}
				if ( isset( $post["gallery_shortcode"] ) ) {
					themify_add_import_action( 'gallery_field', array(
						'id' => $id,
						'fields' => $post["gallery_shortcode"],
					) );
				}
				if ( isset( $post["_product_image_gallery"] ) ) {
					themify_add_import_action( 'product_gallery', array(
						'id' => $id,
						'images' => $post["_product_image_gallery"],
					) );
				}
			}
		}
	}
}
$thumbs = array();
function themify_do_demo_import() {
global $import_actions;

	if ( isset( $GLOBALS["ThemifyBuilder_Data_Manager"] ) ) {
		remove_action( "save_post", array( $GLOBALS["ThemifyBuilder_Data_Manager"], "save_builder_text_only"), 10, 3 );
	}
$term = array (
  'term_id' => 2,
  'name' => 'Main Navigation',
  'slug' => 'main-navigation',
  'term_group' => 0,
  'taxonomy' => 'nav_menu',
  'description' => '',
  'parent' => 0,
);

if( ERASEDEMO ) {
	themify_undo_import_term( $term );
} else {
	themify_import_term( $term );
}

$post = array (
  'ID' => 153,
  'post_date' => '2020-01-10 06:41:31',
  'post_date_gmt' => '2020-01-10 06:41:31',
  'post_content' => '<!-- wp:themify-builder/canvas /--><!--themify_builder_static--><h1>About Us</h1> <p>We want to create a community where everyone can come together and grow. A community of like minded individuals to support each other in their journeys through meditation and mindfulness. </p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/about-us-image-750x838.jpg" width="750" alt="about-us-image" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/01/about-us-image.jpg 750w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/about-us-image-537x600.jpg 537w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/about-us-image-700x782.jpg 700w" sizes="(max-width: 750px) 100vw, 750px" />
<h2>Our Philosophy</h2> <p>We want to present everyone with an opportunity to become a part of our great community. Our community is committed, accepting, and welcoming to experienced and new-coming yoga enthusiasts alike! We specialize in classes anyone of any physical ability and  experience can participate in. We are happy to support everyone through their journeys.</p>
<h2>Our Values</h2>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/health-icon-200x150.png" width="200" title="Health" alt="Health" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/01/health-icon-200x150.png 200w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/health-icon.png 217w" sizes="(max-width: 200px) 100vw, 200px" /> <h3> Health </h3>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/balance-icon-200x150.png" width="200" title="Balance" alt="Balance" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/01/balance-icon-200x150.png 200w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/balance-icon.png 217w" sizes="(max-width: 200px) 100vw, 200px" /> <h3> Balance </h3>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/strength-icon-200x150.png" width="200" title="Strength" alt="Strength" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/01/strength-icon-200x150.png 200w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/strength-icon.png 217w" sizes="(max-width: 200px) 100vw, 200px" /> <h3> Strength </h3>
<h2>Photo Gallery</h2> <p>Every class is different and accessible for students of any level. Come see our instructors providing a creative and well-rounded approach.</p>
<ul data-id="tb_yrb3139" data-visible="3" data-mob-visible="1" data-scroll="1" data-auto-scroll="off" data-speed="1" data-wrap="yes" data-arrow="yes" data-pagination="no" data-effect="scroll" data-height="variable" data-pause-on-hover="resume" data-play-controller="no" data-horizontal="no" > 
 <li> <a href="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-gallery-1.jpg" alt="" data-rel="tb_yrb3139"> <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-gallery-1-365x308.jpg" width="365" height="308" alt="yoga-gallery-1" /> </a> </li> <li> <a href="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-gallery-2.jpg" alt="" data-rel="tb_yrb3139"> <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-gallery-2-365x308.jpg" width="365" height="308" alt="yoga-gallery-2" /> </a> </li> <li> <a href="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-gallery-3.jpg" alt="" data-rel="tb_yrb3139"> <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-gallery-3-365x308.jpg" width="365" height="308" alt="yoga-gallery-3" /> </a> </li> <li> <a href="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-gallery-4.jpg" alt="" data-rel="tb_yrb3139"> <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-gallery-4-365x308.jpg" width="365" height="308" alt="yoga-gallery-4" /> </a> </li> <li> <a href="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-gallery-5.jpg" alt="" data-rel="tb_yrb3139"> <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-gallery-5-365x308.jpg" width="365" height="308" alt="yoga-gallery-5" /> </a> </li> </ul>
<h2>Meet Our Instructors</h2>
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/mariana-360x326.jpg" width="360" title="Maria" alt="Vinyasa Yoga Instructor" /> </a> <h3> <a href="https://themify.me/" > Maria </a> </h3> Vinyasa Yoga Instructor
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/rosie-360x326.jpg" width="360" title="Rose" alt="Hot Yoga Instructor" /> </a> <h3> <a href="https://themify.me/" > Rose </a> </h3> Hot Yoga Instructor
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/julian-360x326.jpg" width="360" title="Juliana" alt="Iyengar Yoga Instructor" /> </a> <h3> <a href="https://themify.me/" > Juliana </a> </h3> Iyengar Yoga Instructor
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/jenny-360x326.jpg" width="360" title="Jeanny" alt="Yin Yoga Instructor" /> </a> <h3> <a href="https://themify.me/" > Jeanny </a> </h3> Yin Yoga Instructor
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/roxane-360x326.jpg" width="360" title="Roxanne" alt="Strength Yoga Instructor" /> </a> <h3> <a href="https://themify.me/" > Roxanne </a> </h3> Strength Yoga Instructor
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/kimberly-360x326.jpg" width="360" title="Kim" alt="Restorative Yoga Instructor" /> </a> <h3> <a href="https://themify.me/" > Kim </a> </h3> Restorative Yoga Instructor<!--/themify_builder_static-->',
  'post_title' => 'About',
  'post_excerpt' => '',
  'post_name' => 'about',
  'post_modified' => '2020-02-28 09:03:52',
  'post_modified_gmt' => '2020-02-28 09:03:52',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?page_id=153',
  'menu_order' => 0,
  'post_type' => 'page',
  'meta_input' => 
  array (
    'page_layout' => 'sidebar-none',
    'content_width' => 'full_width',
    'hide_page_title' => 'yes',
    'section_scrolling_mobile' => 'on',
    'mobile_menu_styles' => 'default',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"zi7c562\\",\\"cols\\":[{\\"element_id\\":\\"p3t2563\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"aord563\\",\\"mod_settings\\":{\\"content_text\\":\\"<h1>About Us<\\\\/h1>\\\\n<p>We want to create a community where everyone can come together and grow. A community of like minded individuals to support each other in their journeys through meditation and mindfulness. <\\\\/p>\\",\\"breakpoint_mobile\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"}}}]},{\\"element_id\\":\\"5h16563\\",\\"grid_class\\":\\"col4-2\\"}],\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":\\"16\\",\\"padding_opp_top\\":false,\\"padding_top\\":\\"12\\",\\"background_position\\":\\"50,46.25\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/about-hero-bg.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"20\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"font_style_h3_regular\\":\\"normal\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"regular\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"70\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"breakpoint_mobile\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"35\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"30\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"20\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"8\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"padding_left\\":\\"0\\",\\"padding_right\\":\\"0\\"},\\"breakpoint_tablet\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"50\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"10\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"16\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"12\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"55\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"15\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"background_gradient-circle-radial\\":false,\\"background_gradient-gradient-angle\\":\\"180\\",\\"background_gradient-gradient-type\\":\\"linear\\"}},{\\"element_id\\":\\"pw48620\\",\\"cols\\":[{\\"element_id\\":\\"wa0u621\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"p6vi621\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":false,\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/about-us-image.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-left\\",\\"breakpoint_mobile\\":{\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"%\\",\\"margin_bottom\\":\\"-20\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"},\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_width_unit\\":\\"px\\",\\"width_image\\":\\"750\\"}}],\\"styling\\":{\\"padding_bottom\\":0,\\"padding_bottom_unit\\":\\"%\\",\\"padding_top\\":0,\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_opp_top\\":\\"1\\"}},{\\"element_id\\":\\"hekx622\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"5qf9622\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Our Philosophy<\\\\/h2>\\\\n<p>We want to present everyone with an opportunity to become a part of our great community. Our community is committed, accepting, and welcoming to experienced and new-coming yoga enthusiasts alike! We specialize in classes anyone of any physical ability and  experience can participate in. We are happy to support everyone through their journeys.<\\\\/p>\\",\\"padding_right_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_right\\":\\"20\\",\\"padding_opp_top\\":false,\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"10\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"10\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column4-2\\",\\"styling\\":{\\"background_position\\":\\"0,100\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat-none\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth-content\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":0,\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":0,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"margin-top_opp_top\\":false,\\"margin-bottom_unit\\":\\"%\\",\\"margin-bottom\\":\\"-5\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"35\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"margin-bottom_unit\\":\\"%\\",\\"margin-bottom\\":\\"8\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"0\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"0\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"},\\"breakpoint_tablet\\":{\\"margin-bottom_unit\\":\\"%\\",\\"margin-bottom\\":\\"10\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"0\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"6\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"45\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"zi\\":\\"3\\",\\"margin-bottom_unit\\":\\"%\\",\\"margin-bottom\\":\\"0\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"margin-top_unit\\":\\"px\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}},{\\"element_id\\":\\"kc8t540\\",\\"cols\\":[{\\"element_id\\":\\"bx6h540\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"9knu633\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Our Values<\\\\/h2>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"40\\",\\"margin_opp_top\\":false,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_color\\":\\"#000000\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"35\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"h4_margin_bottom_unit\\":\\"px\\",\\"h4_margin_top_unit\\":\\"px\\",\\"t_shh4_blur_unit\\":\\"px\\",\\"t_shh4_vShadow_unit\\":\\"px\\",\\"t_shh4_hShadow_unit\\":\\"px\\",\\"letter_spacing_h4_unit\\":\\"px\\",\\"line_height_h4_unit\\":\\"px\\",\\"font_size_h4_unit\\":\\"px\\",\\"font_gradient_color_h4-circle-radial\\":false,\\"font_gradient_color_h4-gradient-angle\\":\\"180\\",\\"font_gradient_color_h4-gradient-type\\":\\"linear\\",\\"font_color_type_h4\\":\\"font_color_h4_solid\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"45\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"},\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\",\\"font_gradient_color-circle-radial\\":false,\\"font_gradient_color-gradient-angle\\":\\"180\\",\\"font_gradient_color-gradient-type\\":\\"linear\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"}},{\\"element_id\\":\\"akqy990\\",\\"cols\\":[{\\"element_id\\":\\"bko0992\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"uekj439\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Health\\",\\"auto_fullwidth\\":false,\\"width_image\\":\\"200\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/health-icon-200x150.png\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\"}}]},{\\"element_id\\":\\"r424992\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"g1mh400\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Balance\\",\\"auto_fullwidth\\":false,\\"width_image\\":\\"200\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/balance-icon-200x150.png\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\"}}]},{\\"element_id\\":\\"tusx992\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"ancz193\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Strength\\",\\"auto_fullwidth\\":false,\\"width_image\\":\\"200\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/strength-icon-200x150.png\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"col_tablet\\":\\"column3-1\\"}]}],\\"styling\\":{\\"hide_anchor\\":false,\\"text_align\\":\\"center\\",\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"4\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"text_align\\":\\"center\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_size_h2\\":\\"55\\"}},{\\"element_id\\":\\"hseq828\\",\\"cols\\":[{\\"element_id\\":\\"ywvn829\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"iz43830\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Photo Gallery<\\\\/h2>\\\\n<p>Every class is different and accessible for students of any level. Come see our instructors providing a creative and well-rounded approach.<\\\\/p>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"40\\",\\"margin_opp_top\\":false,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_color\\":\\"#000000\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"35\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"font_style_h3_regular\\":\\"italic\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"45\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\",\\"font_color\\":\\"#000000\\",\\"font_gradient_color-circle-radial\\":false,\\"font_gradient_color-gradient-angle\\":\\"180\\",\\"font_gradient_color-gradient-type\\":\\"linear\\",\\"font_color_type\\":\\"font_color_solid\\"},\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\",\\"font_gradient_color-circle-radial\\":false,\\"font_gradient_color-gradient-angle\\":\\"180\\",\\"font_gradient_color-gradient-type\\":\\"linear\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"}},{\\"mod_name\\":\\"slider\\",\\"element_id\\":\\"yrb3139\\",\\"mod_settings\\":{\\"posts_per_page_slider\\":\\"4\\",\\"display_slider\\":\\"none\\",\\"img_w_slider\\":\\"365\\",\\"img_h_slider\\":\\"308\\",\\"horizontal\\":\\"no\\",\\"visible_opt_slider\\":\\"3\\",\\"pause_on_hover_slider\\":\\"resume\\",\\"play_pause_control\\":\\"no\\",\\"show_arrow_slider\\":\\"yes\\",\\"show_nav_slider\\":\\"no\\",\\"wrap_slider\\":\\"yes\\",\\"auto_scroll_opt_slider\\":\\"off\\",\\"post_type\\":\\"post\\",\\"height_slider\\":\\"variable\\",\\"show_arrow_buttons_vertical\\":false,\\"effect_slider\\":\\"scroll\\",\\"speed_opt_slider\\":\\"normal\\",\\"scroll_opt_slider\\":\\"1\\",\\"mob_visible_opt_slider\\":\\"1\\",\\"img_fullwidth_slider\\":false,\\"layout_slider\\":\\"slider-default\\",\\"img_content_slider\\":[{\\"img_url_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-gallery-1.jpg\\",\\"img_link_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-gallery-1.jpg\\",\\"img_link_params\\":\\"lightbox\\"},{\\"img_url_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-gallery-2.jpg\\",\\"img_link_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-gallery-2.jpg\\",\\"img_link_params\\":\\"lightbox\\"},{\\"img_url_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-gallery-3.jpg\\",\\"img_link_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-gallery-3.jpg\\",\\"img_link_params\\":\\"lightbox\\"},{\\"img_url_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-gallery-4.jpg\\",\\"img_link_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-gallery-4.jpg\\",\\"img_link_params\\":\\"lightbox\\"},{\\"img_url_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-gallery-5.jpg\\",\\"img_link_slider\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-gallery-5.jpg\\",\\"img_link_params\\":\\"lightbox\\"}],\\"open_link_new_tab_slider\\":\\"no\\",\\"unlink_feat_img_slider\\":\\"no\\",\\"hide_feat_img_slider\\":\\"no\\",\\"unlink_post_title_slider\\":\\"no\\",\\"hide_post_title_slider\\":\\"no\\",\\"orderby_slider\\":\\"date\\",\\"order_slider\\":\\"desc\\",\\"blog_category_slider\\":\\"0|single\\",\\"taxonomy\\":\\"category\\",\\"layout_display_slider\\":\\"image\\",\\"checkbox_i_r_c_apply_all\\":\\"1\\",\\"i_r_c_left_unit\\":\\"px\\",\\"i_r_c_opp_left\\":false,\\"i_r_c_right_unit\\":\\"px\\",\\"i_r_c_bottom_unit\\":\\"px\\",\\"i_r_c_opp_bottom\\":false,\\"i_r_c_top_unit\\":\\"px\\",\\"i_r_c_top\\":\\"8\\",\\"breakpoint_tablet\\":{\\"checkbox_i_m_apply_all\\":false,\\"i_m_left_unit\\":\\"px\\",\\"i_m_opp_left\\":\\"1\\",\\"i_m_bottom_unit\\":\\"px\\",\\"i_m_right_unit\\":\\"px\\",\\"i_m_opp_top\\":false,\\"i_m_top_unit\\":\\"px\\"}}}]}],\\"styling\\":{\\"hide_anchor\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":4,\\"padding_opp_top\\":false,\\"padding_top\\":8,\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"0\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"8\\"},\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}},{\\"element_id\\":\\"59zb632\\",\\"cols\\":[{\\"element_id\\":\\"q2sm632\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"flo07\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Meet Our Instructors<\\\\/h2>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"40\\",\\"margin_opp_top\\":false,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#000000\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"35\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"text_align\\":\\"center\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\",\\"font_color\\":\\"#000000\\",\\"font_gradient_color-circle-radial\\":false,\\"font_gradient_color-gradient-angle\\":\\"180\\",\\"font_gradient_color-gradient-type\\":\\"linear\\",\\"font_color_type\\":\\"font_color_solid\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"45\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\"}},{\\"element_id\\":\\"6pfi634\\",\\"cols\\":[{\\"element_id\\":\\"g3re634\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"pgxj635\\",\\"mod_settings\\":{\\"caption_image\\":\\"Vinyasa Yoga Instructor\\",\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Maria\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/mariana-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"i_t_r_c_opp_left\\":\\"1\\",\\"i_t_r_c_opp_bottom\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":\\"1\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\",\\"i_t_m_opp_left\\":false,\\"i_t_m_opp_top\\":false}},{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"wtda635\\",\\"mod_settings\\":{\\"caption_image\\":\\"Hot Yoga Instructor\\",\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Rose\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/rosie-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"i_t_r_c_opp_left\\":\\"1\\",\\"i_t_r_c_opp_bottom\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":\\"1\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}}]},{\\"element_id\\":\\"opgo635\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"zvis635\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/julian-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"caption_image\\":\\"Iyengar Yoga Instructor\\",\\"title_image\\":\\"Juliana\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}},{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"ar64635\\",\\"mod_settings\\":{\\"caption_image\\":\\"Yin Yoga Instructor\\",\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Jeanny\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/jenny-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"i_t_r_c_opp_left\\":\\"1\\",\\"i_t_r_c_opp_bottom\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":\\"1\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}}]},{\\"element_id\\":\\"8lbp636\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"swty636\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/roxane-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"caption_image\\":\\"Strength Yoga Instructor\\",\\"title_image\\":\\"Roxanne\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}},{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"z3fm636\\",\\"mod_settings\\":{\\"caption_image\\":\\"Restorative Yoga Instructor\\",\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Kim\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/kimberly-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"i_t_r_c_opp_left\\":\\"1\\",\\"i_t_r_c_opp_bottom\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":\\"1\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}}]}],\\"col_tablet_landscape\\":\\"column3-1\\",\\"col_tablet\\":\\"column3-1\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"0\\",\\"margin_opp_top\\":false,\\"checkbox_padding_apply_all\\":\\"1\\",\\"padding_opp_left\\":false,\\"padding_opp_top\\":false}}]}],\\"styling\\":{\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":\\"5\\",\\"padding_opp_top\\":false,\\"padding_top\\":\\"0\\",\\"f_g_c_h3_h-circle-radial\\":false,\\"f_c_t_h3_h\\":\\"f_c_h3_h_solid\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"4\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"6\\"},\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"breakpoint_tablet\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"5\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"0\\"},\\"breakpoint_tablet_landscape\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"5\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"0\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}}]',
    'themify_used_global_styles' => 
    array (
      0 => 'tb_gs415e1',
    ),
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 159,
  'post_date' => '2020-01-10 07:07:37',
  'post_date_gmt' => '2020-01-10 07:07:37',
  'post_content' => '<!-- wp:themify-builder/canvas /--><!--themify_builder_static--><h1>Classes</h1> <p>From quiet meditation to strength training and everything in between. We are here to support you through your journey.</p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/02/asian-doing-yoga-360x389.jpg" width="360" alt="asian-doing-yoga" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/02/asian-doing-yoga-360x389.jpg 360w, https://themify.me/demo/themes/ultra-yoga/files/2020/02/asian-doing-yoga.jpg 400w" sizes="(max-width: 360px) 100vw, 360px" />
<h3>Yin Yoga</h3> <p>All Levels Welcome Beginner Friendly</p>
<a href="https://themify.me/" > Sign Up </a>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/02/yoga-partner-360x389.jpg" width="360" alt="yoga-partner" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/02/yoga-partner-360x389.jpg 360w, https://themify.me/demo/themes/ultra-yoga/files/2020/02/yoga-partner.jpg 400w" sizes="(max-width: 360px) 100vw, 360px" />
<h3>Iyengar Yoga</h3> <p>All Levels Welcome Beginner Friendly</p>
<a href="https://themify.me/" > Sign Up </a>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/02/asian-yoga-360x389.jpg" width="360" alt="asian-yoga" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/02/asian-yoga-360x389.jpg 360w, https://themify.me/demo/themes/ultra-yoga/files/2020/02/asian-yoga.jpg 400w" sizes="(max-width: 360px) 100vw, 360px" />
<h3>Strength Yoga</h3> <p>All Levels Welcome Beginner Friendly</p>
<a href="https://themify.me/" > Sign Up </a>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/02/yoga-strecthing-360x389.jpg" width="360" alt="yoga-strecthing" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/02/yoga-strecthing-360x389.jpg 360w, https://themify.me/demo/themes/ultra-yoga/files/2020/02/yoga-strecthing.jpg 400w" sizes="(max-width: 360px) 100vw, 360px" />
<h3>Hot Yoga</h3> <p>All Levels Welcome Beginner Friendly</p>
<a href="https://themify.me/" > Sign Up </a>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/02/woman-doing-yoga-360x389.jpg" width="360" alt="woman-doing-yoga" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/02/woman-doing-yoga-360x389.jpg 360w, https://themify.me/demo/themes/ultra-yoga/files/2020/02/woman-doing-yoga.jpg 400w" sizes="(max-width: 360px) 100vw, 360px" />
<h3>Restorative Yoga</h3> <p>All Levels Welcome Beginner Friendly</p>
<a href="https://themify.me/" > Sign Up </a>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/02/man-doing-yoga-360x389.jpg" width="360" alt="man-doing-yoga" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/02/man-doing-yoga-360x389.jpg 360w, https://themify.me/demo/themes/ultra-yoga/files/2020/02/man-doing-yoga.jpg 400w" sizes="(max-width: 360px) 100vw, 360px" />
<h3>Vinyasa Yoga</h3> <p>All Levels Welcome Beginner Friendly</p>
<a href="https://themify.me/" > Sign Up </a><!--/themify_builder_static-->',
  'post_title' => 'Classes',
  'post_excerpt' => '',
  'post_name' => 'classes',
  'post_modified' => '2020-02-28 09:04:30',
  'post_modified_gmt' => '2020-02-28 09:04:30',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?page_id=159',
  'menu_order' => 0,
  'post_type' => 'page',
  'meta_input' => 
  array (
    'page_layout' => 'sidebar-none',
    'content_width' => 'full_width',
    'hide_page_title' => 'yes',
    'section_scrolling_mobile' => 'on',
    'mobile_menu_styles' => 'default',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"ihtx590\\",\\"cols\\":[{\\"element_id\\":\\"79fb591\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"zkhc591\\",\\"mod_settings\\":{\\"content_text\\":\\"<h1>Classes<\\\\/h1>\\\\n<p>From quiet meditation to strength training and everything in between. We are here to support you through your journey.<\\\\/p>\\",\\"breakpoint_mobile\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"}}}]},{\\"element_id\\":\\"js70593\\",\\"grid_class\\":\\"col4-2\\"}],\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column4-3-4-1\\",\\"styling\\":{\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":\\"16\\",\\"padding_opp_top\\":false,\\"padding_top\\":\\"12\\",\\"background_position\\":\\"50,46.25\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/classes-hero-bg.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"20\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"font_style_h3_regular\\":\\"normal\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"regular\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"70\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"breakpoint_mobile\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"45\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"30\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"20\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"8\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"breakpoint_tablet\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"60\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"10\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"60\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"15\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"background_gradient-circle-radial\\":false,\\"background_gradient-gradient-angle\\":\\"180\\",\\"background_gradient-gradient-type\\":\\"linear\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"background_color\\":\\"#eceffe\\",\\"padding_right\\":0,\\"cover_gradient-gradient\\":\\"0% rgb(255, 255, 255)|100% rgba(255, 255, 255, 0)\\",\\"cover_gradient-circle-radial\\":false,\\"cover_gradient-gradient-angle\\":\\"99\\",\\"cover_gradient-gradient-type\\":\\"linear\\",\\"cover_color-type\\":\\"cover_gradient\\"}},{\\"element_id\\":\\"27lp559\\",\\"cols\\":[{\\"element_id\\":\\"x18y560\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"element_id\\":\\"uzuq638\\",\\"cols\\":[{\\"element_id\\":\\"1msa638\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"3fqy639\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/asian-doing-yoga.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"breakpoint_mobile\\":{\\"custom_parallax_scroll_zindex\\":\\"2\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"},\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_width_unit\\":\\"px\\"}}],\\"grid_width\\":66},{\\"element_id\\":\\"6zpm639\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"o0bl639\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Yin Yoga<\\\\/h3>\\\\n<p>All Levels Welcome Beginner Friendly<\\\\/p>\\",\\"global_styles\\":\\"tb_gs7795e5\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"}}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"46vh639\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Sign Up\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"button_color_bg\\":\\"purple\\",\\"icon_alignment\\":\\"left\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\",\\"global_styles\\":\\"tb_gs3285e2\\",\\"alignment\\":\\"center\\"}}],\\"grid_width\\":34}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"30\\",\\"margin_opp_top\\":false}}]},{\\"element_id\\":\\"ko05560\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"element_id\\":\\"rt20734\\",\\"cols\\":[{\\"element_id\\":\\"yyc5735\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"eqa1736\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/yoga-partner-360x389.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"breakpoint_mobile\\":{\\"custom_parallax_scroll_zindex\\":\\"2\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"},\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_width_unit\\":\\"px\\"}}],\\"grid_width\\":66},{\\"element_id\\":\\"ylpr737\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"egql737\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Iyengar Yoga<\\\\/h3>\\\\n<p>All Levels Welcome Beginner Friendly<\\\\/p>\\",\\"global_styles\\":\\"tb_gs7795e5\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"}}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"wmy8737\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Sign Up\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"button_color_bg\\":\\"purple\\",\\"icon_alignment\\":\\"left\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\",\\"global_styles\\":\\"tb_gs3285e2\\",\\"alignment\\":\\"center\\"}}],\\"grid_width\\":34}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"30\\",\\"margin_opp_top\\":false}}]}],\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column-full\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"padding_top\\":9,\\"padding_top_unit\\":\\"%\\",\\"hide_anchor\\":false}},{\\"element_id\\":\\"gywl846\\",\\"cols\\":[{\\"element_id\\":\\"vk3l846\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"element_id\\":\\"hqu3890\\",\\"cols\\":[{\\"element_id\\":\\"8vfn890\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"elgw890\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/asian-yoga-360x389.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"breakpoint_mobile\\":{\\"custom_parallax_scroll_zindex\\":\\"2\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"},\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_width_unit\\":\\"px\\"}}]},{\\"element_id\\":\\"bcks891\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"5d6w891\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Strength Yoga<\\\\/h3>\\\\n<p>All Levels Welcome Beginner Friendly<\\\\/p>\\",\\"global_styles\\":\\"tb_gs7795e5\\"}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"g7ev891\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Sign Up\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"button_color_bg\\":\\"purple\\",\\"icon_alignment\\":\\"left\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\",\\"l_b_r_c_opp_left\\":false,\\"l_b_r_c_opp_bottom\\":false,\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"checkbox_link_margin_apply_all\\":false,\\"link_margin_left_unit\\":\\"px\\",\\"link_margin_opp_left\\":false,\\"link_margin_bottom_unit\\":\\"px\\",\\"link_margin_right_unit\\":\\"px\\",\\"link_margin_opp_top\\":false,\\"link_margin_top_unit\\":\\"px\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"font_style_regular\\":\\"normal\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\",\\"font_gradient_color-circle-radial\\":false,\\"font_gradient_color-gradient-angle\\":\\"180\\",\\"font_gradient_color-gradient-type\\":\\"linear\\",\\"font_color_type\\":\\"font_color_solid\\",\\"global_styles\\":\\"tb_gs3285e2\\",\\"alignment\\":\\"center\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"30\\",\\"margin_opp_top\\":false}}]},{\\"element_id\\":\\"ntyd847\\",\\"grid_class\\":\\"col2-1\\",\\"modules\\":[{\\"element_id\\":\\"75u2593\\",\\"cols\\":[{\\"element_id\\":\\"nar8593\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"wsz8593\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/yoga-strecthing-360x389.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"breakpoint_mobile\\":{\\"custom_parallax_scroll_zindex\\":\\"2\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"},\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_width_unit\\":\\"px\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"}}]},{\\"element_id\\":\\"vcrb594\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"fiqm594\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Hot Yoga<\\\\/h3>\\\\n<p>All Levels Welcome Beginner Friendly<\\\\/p>\\",\\"global_styles\\":\\"tb_gs7795e5\\"}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"lr6a594\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Sign Up\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"button_color_bg\\":\\"purple\\",\\"icon_alignment\\":\\"left\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\",\\"global_styles\\":\\"tb_gs3285e2\\",\\"alignment\\":\\"center\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"30\\",\\"margin_opp_top\\":false}}]}],\\"col_tablet\\":\\"column-full\\",\\"styling\\":{\\"padding_top\\":30,\\"padding_top_unit\\":\\"px\\"}},{\\"element_id\\":\\"bv4t898\\",\\"cols\\":[{\\"element_id\\":\\"ud2a900\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"element_id\\":\\"fywo96\\",\\"cols\\":[{\\"element_id\\":\\"8s5u101\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"f05h549\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/woman-doing-yoga-360x389.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"breakpoint_mobile\\":{\\"custom_parallax_scroll_zindex\\":\\"2\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"},\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_width_unit\\":\\"px\\"}}]},{\\"element_id\\":\\"i5c3102\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"gvry882\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Restorative Yoga<\\\\/h3>\\\\n<p>All Levels Welcome Beginner Friendly<\\\\/p>\\",\\"global_styles\\":\\"tb_gs7795e5\\"}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"b7tn54\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Sign Up\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"button_color_bg\\":\\"purple\\",\\"icon_alignment\\":\\"left\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\",\\"l_b_r_c_opp_left\\":false,\\"l_b_r_c_opp_bottom\\":false,\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"checkbox_link_margin_apply_all\\":false,\\"link_margin_left_unit\\":\\"px\\",\\"link_margin_opp_left\\":false,\\"link_margin_bottom_unit\\":\\"px\\",\\"link_margin_right_unit\\":\\"px\\",\\"link_margin_opp_top\\":false,\\"link_margin_top_unit\\":\\"px\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"font_style_regular\\":\\"normal\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\",\\"font_gradient_color-circle-radial\\":false,\\"font_gradient_color-gradient-angle\\":\\"180\\",\\"font_gradient_color-gradient-type\\":\\"linear\\",\\"font_color_type\\":\\"font_color_solid\\",\\"global_styles\\":\\"tb_gs3285e2\\",\\"alignment\\":\\"center\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"30\\",\\"margin_opp_top\\":false,\\"breakpoint_mobile\\":{\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"30\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_bottom\\":\\"30\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"}}}]},{\\"element_id\\":\\"tprx737\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"element_id\\":\\"eia0474\\",\\"cols\\":[{\\"element_id\\":\\"aiqg481\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"ujof160\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/man-doing-yoga-360x389.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"breakpoint_mobile\\":{\\"custom_parallax_scroll_zindex\\":\\"2\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"},\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_width_unit\\":\\"px\\"}}]},{\\"element_id\\":\\"2esz483\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"cqp4205\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Vinyasa Yoga<\\\\/h3>\\\\n<p>All Levels Welcome Beginner Friendly<\\\\/p>\\",\\"global_styles\\":\\"tb_gs7795e5\\"}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"hyxh604\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Sign Up\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"button_color_bg\\":\\"purple\\",\\"icon_alignment\\":\\"left\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\",\\"global_styles\\":\\"tb_gs3285e2\\",\\"alignment\\":\\"center\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"styling\\":{\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"30\\",\\"margin_opp_top\\":false}}]}],\\"col_tablet\\":\\"column-full\\",\\"styling\\":{\\"padding_top_unit\\":\\"px\\",\\"hide_anchor\\":false,\\"padding_opp_left\\":false,\\"padding_opp_top\\":false,\\"padding_top\\":30}},{\\"element_id\\":\\"5rtw932\\",\\"cols\\":[{\\"element_id\\":\\"x6fb935\\",\\"grid_class\\":\\"col4-2\\"},{\\"element_id\\":\\"wgmh441\\",\\"grid_class\\":\\"col4-2\\"}],\\"col_tablet\\":\\"column4-2\\",\\"styling\\":{\\"text_align\\":\\"center\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_opp_top\\":\\"1\\",\\"hide_anchor\\":false,\\"margin-top_opp_top\\":false}}]',
    'themify_used_global_styles' => 
    array (
      0 => 'tb_gs7795e5',
      1 => 'tb_gs3285e2',
    ),
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 167,
  'post_date' => '2020-01-10 07:08:49',
  'post_date_gmt' => '2020-01-10 07:08:49',
  'post_content' => '<!-- wp:themify-builder/canvas /--><!--themify_builder_static--><h1>Contact</h1> <p>We\'re always here for you. Anytime. Anywhere.</p>
<h2>Get in Touch</h2> <p>Have any questions about our classes, instructors or memberships? Drop in or contact us below.</p>
<form action="https://themify.me/demo/themes/ultra-yoga/wp-admin/admin-ajax.php" id="contact-0--form" method="post"> <label for="contact-0--contact-name">Name *</label> <input type="text" name="contact-name" placeholder="" id="contact-0--contact-name" value="" required/> <label for="contact-0--contact-email">Email *</label> <input type="text" name="contact-email" placeholder="" id="contact-0--contact-email" value="" required /> <label for="contact-0--contact-subject">Subject *</label> <input type="text" name="contact-subject" placeholder="" id="contact-0--contact-subject" value="" required /> <label for="contact-0--contact-message">Message *</label> <textarea name="contact-message" placeholder="" id="contact-0--contact-message" rows="8" cols="45" required></textarea> <button type="submit"> Send </button> </form>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/map-studio.jpg" alt="map-studio" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/01/map-studio.jpg 1500w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/map-studio-600x255.jpg 600w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/map-studio-768x327.jpg 768w" sizes="(max-width: 1500px) 100vw, 1500px" /><!--/themify_builder_static-->',
  'post_title' => 'Contact',
  'post_excerpt' => '',
  'post_name' => 'contact',
  'post_modified' => '2020-02-26 15:08:31',
  'post_modified_gmt' => '2020-02-26 15:08:31',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?page_id=167',
  'menu_order' => 0,
  'post_type' => 'page',
  'meta_input' => 
  array (
    'page_layout' => 'sidebar-none',
    'content_width' => 'full_width',
    'hide_page_title' => 'yes',
    'section_scrolling_mobile' => 'on',
    'mobile_menu_styles' => 'default',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"nb1m697\\",\\"cols\\":[{\\"element_id\\":\\"enyh697\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"1x7m697\\",\\"mod_settings\\":{\\"content_text\\":\\"<h1>Contact<\\\\/h1>\\\\n<p>We\\\'re always here for you. Anytime. Anywhere.<\\\\/p>\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}}]},{\\"element_id\\":\\"m9wo968\\",\\"grid_class\\":\\"col4-2\\"}],\\"col_tablet\\":\\"column3-2-3-1\\",\\"col_mobile\\":\\"column3-2-3-1\\",\\"styling\\":{\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-location-hero-banner.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":\\"23\\",\\"padding_opp_top\\":false,\\"padding_top\\":\\"12\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"20\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"cover_color\\":\\"#ffffff_0.24\\",\\"cover_color-type\\":\\"color\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"70\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"23\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"12\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"45\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"10\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"breakpoint_tablet\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"60\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"23\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"12\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"15\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"breakpoint_tablet_landscape\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"65\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"17\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\"}},{\\"element_id\\":\\"lvw0423\\",\\"cols\\":[{\\"element_id\\":\\"rj5f424\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"fo0m25\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Get in Touch<\\\\/h2>\\\\n<p>Have any questions about our classes, instructors or memberships? Drop in or contact us below.<\\\\/p>\\",\\"text_align\\":\\"center\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"},\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\"}},{\\"mod_name\\":\\"contact\\",\\"element_id\\":\\"tbup48\\",\\"mod_settings\\":{\\"field_name_label\\":\\"Name\\",\\"field_email_label\\":\\"Email\\",\\"field_subject_label\\":\\"Subject\\",\\"field_message_label\\":\\"Message\\",\\"field_sendcopy_label\\":\\"Send a copy to myself\\",\\"field_send_label\\":\\"Send\\",\\"gdpr_label\\":\\"I consent to my submitted data being collected and stored\\",\\"field_name_require\\":\\"yes\\",\\"field_email_require\\":\\"yes\\",\\"field_name_active\\":\\"yes\\",\\"field_email_active\\":\\"yes\\",\\"field_subject_active\\":\\"yes\\",\\"field_subject_require\\":\\"yes\\",\\"field_message_active\\":\\"yes\\",\\"field_send_align\\":\\"right\\",\\"field_extra\\":\\"{ \\\\\\"fields\\\\\\": [] }\\",\\"field_order\\":\\"{}\\",\\"field_optin_label\\":\\"Subscribe to my newsletter.\\",\\"field_optin_active\\":false,\\"provider\\":\\"mailchimp\\",\\"field_sendcopy_active\\":false,\\"field_captcha_active\\":false,\\"contact_sent_from\\":\\"enable\\",\\"post_author\\":false,\\"user_role\\":\\"admin\\",\\"send_to_admins\\":false,\\"layout_contact\\":\\"animated-label\\"}}],\\"styling\\":{\\"b_sh_inset\\":false,\\"b_sh_color\\":\\"#000000_0.10\\",\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_spread\\":\\"0\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_blur\\":\\"12\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_vOffset\\":\\"3\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"b_sh_hOffset\\":\\"0\\",\\"checkbox_b_ra_apply_all\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"b_ra_top\\":\\"8\\",\\"border-type\\":\\"top\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_right_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"7\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom\\":\\"6\\",\\"padding_right\\":\\"7\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"6\\",\\"background_color\\":\\"#ffffff\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"margin-top_opp_top\\":false}}],\\"styling\\":{\\"padding_opp_left\\":false,\\"padding_opp_top\\":false,\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"%\\",\\"margin-top\\":\\"-15\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"zi\\":\\"3\\",\\"hide_anchor\\":false,\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"%\\",\\"margin-top\\":\\"-15\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"35\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"},\\"breakpoint_tablet\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"%\\",\\"margin-top\\":\\"-18\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"45\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"},\\"breakpoint_tablet_landscape\\":{\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"50\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"%\\",\\"margin-top\\":\\"-18\\"},\\"margin-bottom_unit\\":\\"px\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_size_h2\\":\\"55\\"}},{\\"element_id\\":\\"whis209\\",\\"cols\\":[{\\"element_id\\":\\"b2xo211\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"hk00386\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/map-studio.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-top\\",\\"margin_opp_left\\":false,\\"margin_opp_top\\":false}}]}],\\"styling\\":{\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth-content\\",\\"zi\\":\\"1\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"%\\",\\"margin-top\\":\\"-10\\"}}]',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 5,
  'post_date' => '2020-01-09 04:23:02',
  'post_date_gmt' => '2020-01-09 04:23:02',
  'post_content' => '<!-- wp:themify-builder/canvas /--><!--themify_builder_static--><h1>Welcome</h1> <p>We want to provide an authentic experience that incorporates original traditional yoga practices in our classes. Our community consists of teachers and students dedicated to offering an elevated experience!</p>
<a href="https://themify.me/" > Book now </a>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/about-us-image-700x782.jpg" width="700" alt="about-us-image" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/01/about-us-image-700x782.jpg 700w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/about-us-image-537x600.jpg 537w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/about-us-image.jpg 750w" sizes="(max-width: 700px) 100vw, 700px" />
<h2>About Us</h2> <p>Our mission is to provide a retreat for everyone to step away from their busy lives and practice yoga within a peaceful and quiet space, while offering a variety of styles to meet the needs of everyone.</p>
<a href="https://themify.me/" > Learn more </a>
<h2>Classes</h2> <p>Our certified teachers aim to provide students with access to a variety of techniques such as Vinyasa, Iyengar, Restorative meditation and many more. A variety of these techniques may be used in any given class or may be elaborated on in special-focus classes.</p> <p>We offer different levels of physical intensity, and different levels of experiance.</p>
<a href="https://themify.me/" > Learn more </a>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/classes-image-700x876.jpg" width="700" alt="classes-image" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/01/classes-image-700x876.jpg 700w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/classes-image-479x600.jpg 479w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/classes-image.jpg 750w" sizes="(max-width: 700px) 100vw, 700px" />
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-pose-a-360x326.jpg" width="360" title="Strength" alt="All Skill Levels Welcome" /> </a> <h3> <a href="https://themify.me/" > Strength </a> </h3> All Skill Levels Welcome
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-pose-d-360x326.jpg" width="360" title="Yin Yoga" alt="All Skill Levels Welcome" /> </a> <h3> <a href="https://themify.me/" > Yin Yoga </a> </h3> All Skill Levels Welcome
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-pose-b-360x325.jpg" width="360" title="Vinyasa" alt="All Skill Levels Welcome" /> </a> <h3> <a href="https://themify.me/" > Vinyasa </a> </h3> All Skill Levels Welcome
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-pose-e-360x325.jpg" width="360" title="Iyengar" alt="All Skill Levels Welcome" /> </a> <h3> <a href="https://themify.me/" > Iyengar </a> </h3> All Skill Levels Welcome
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-pose-c-360x326.jpg" width="360" title="Hot Yoga" alt="All Skill Levels Welcome" /> </a> <h3> <a href="https://themify.me/" > Hot Yoga </a> </h3> All Skill Levels Welcome
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-pose-f-360x326.jpg" width="360" title="Restorative" alt="All Skill Levels Welcome" /> </a> <h3> <a href="https://themify.me/" > Restorative </a> </h3> All Skill Levels Welcome
<a href="https://themify.me/demo/themes/ultra-yoga/classes/" > More Classes </a>
<h2>Pricing</h2>
<h3>Memberships</h3> <p>Unlimited access for as low as $150 a month. 3 month minimum commitment. (Save over 30% on unlimited yoga classes.)</p> <p>$150 Monthly</p>
<a href="https://themify.me/" > Take class </a>
<h3>Drop-In</h3> <p>Book in advance or just drop in. Any class. Ant time.</p> <p>$25 Drop-in classes<br />$100 5-class package<br />$180 10-class package</p>
<a href="https://themify.me/" > Take class </a>
<h4>Addons</h4> <p>Unlimited access to mat and towel services every class.</p> <p>$20 Additional</p>
<a href="https://themify.me/" > Take class </a>
<h2>Instructors</h2>
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/mariana-360x326.jpg" width="360" title="Maria" alt="Vinyasa Yoga Instructor" /> </a> <h3> <a href="https://themify.me/" > Maria </a> </h3> Vinyasa Yoga Instructor
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/julian-360x326.jpg" width="360" title="Juliana" alt="Iyengar Yoga Instructor" /> </a> <h3> <a href="https://themify.me/" > Juliana </a> </h3> Iyengar Yoga Instructor
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/roxane-360x326.jpg" width="360" title="Roxanne" alt="Strength Yoga Instructor" /> </a> <h3> <a href="https://themify.me/" > Roxanne </a> </h3> Strength Yoga Instructor
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/rosie-360x326.jpg" width="360" title="Rose" alt="Hot Yoga Instructor" /> </a> <h3> <a href="https://themify.me/" > Rose </a> </h3> Hot Yoga Instructor
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/jenny-360x326.jpg" width="360" title="Jeanny" alt="Yin Yoga Instructor" /> </a> <h3> <a href="https://themify.me/" > Jeanny </a> </h3> Yin Yoga Instructor
<a href="https://themify.me/" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/kimberly-360x326.jpg" width="360" title="Kim" alt="Restorative Yoga Instructor" /> </a> <h3> <a href="https://themify.me/" > Kim </a> </h3> Restorative Yoga Instructor
<h2>Transform your body, soul, and life today</h2>
<a href="https://www.youtube.com/watch?v=NKiGZktresA" data-zoom-config="1200px|500px" > <img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-image-video-630x517.jpg" width="630" alt="yoga-image-video" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-image-video-630x517.jpg 630w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-image-video-600x493.jpg 600w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-image-video.jpg 632w" sizes="(max-width: 630px) 100vw, 630px" /> </a>
<h2>Locations</h2>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-location-a-353x309.jpg" width="353" alt="yoga-location-a" />
<h3>Bloor</h3> <p>604-676-3941</p> <p>888 Bloor Street, Toronto, Canada, M4Y 1L5</p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-location-b-353x309.jpg" width="353" alt="yoga-location-b" />
<h3>Queen</h3> <p>604-725-4731</p> <p>521 Queen Street, Toronto, Canada, M3N 9B4</p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/map-595x582.jpg" width="595" alt="map" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/01/map-595x582.jpg 595w, https://themify.me/demo/themes/ultra-yoga/files/2020/01/map.jpg 599w" sizes="(max-width: 595px) 100vw, 595px" />
<h2>Contact Us</h2> <p>Have any questions about our classes, instructors, or memberships? Drop in or send us a message!</p>
<form action="https://themify.me/demo/themes/ultra-yoga/wp-admin/admin-ajax.php" id="contact-0--form" method="post"> <label for="contact-0--contact-name">Name *</label> <input type="text" name="contact-name" placeholder="" id="contact-0--contact-name" value="" required/> <label for="contact-0--contact-email">Email *</label> <input type="text" name="contact-email" placeholder="" id="contact-0--contact-email" value="" required /> <label for="contact-0--contact-subject">Subject *</label> <input type="text" name="contact-subject" placeholder="" id="contact-0--contact-subject" value="" required /> <label for="contact-0--contact-message">Message *</label> <textarea name="contact-message" placeholder="" id="contact-0--contact-message" rows="8" cols="45" required></textarea> <button type="submit"> Send </button> </form><!--/themify_builder_static-->',
  'post_title' => 'Home',
  'post_excerpt' => '',
  'post_name' => 'home',
  'post_modified' => '2020-03-05 04:11:11',
  'post_modified_gmt' => '2020-03-05 04:11:11',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?page_id=5',
  'menu_order' => 0,
  'post_type' => 'page',
  'meta_input' => 
  array (
    'page_layout' => 'sidebar-none',
    'content_width' => 'full_width',
    'hide_page_title' => 'yes',
    'section_scrolling_mobile' => 'on',
    'mobile_menu_styles' => 'default',
    'header_wrap' => 'transparent',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"5962345\\",\\"cols\\":[{\\"element_id\\":\\"r9be346\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"kkna921\\",\\"mod_settings\\":{\\"content_text\\":\\"<h1>Welcome<\\\\/h1>\\\\n<p>We want to provide an authentic experience that incorporates original traditional yoga practices in our classes. Our community consists of teachers and students dedicated to offering an elevated experience!<\\\\/p>\\",\\"breakpoint_mobile\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"}}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"mdjv979\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Book now\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"button_color_bg\\":\\"purple\\",\\"icon_alignment\\":\\"left\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\"}}]},{\\"element_id\\":\\"hyh2523\\",\\"grid_class\\":\\"col4-2\\"}],\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column4-2\\",\\"styling\\":{\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":\\"22\\",\\"padding_opp_top\\":false,\\"padding_top\\":\\"20\\",\\"background_position\\":\\"50,46.25\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/home-hero-bg.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"20\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"font_style_h3_regular\\":\\"normal\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"70\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"breakpoint_mobile\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"45\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"30\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"20\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"8\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"breakpoint_tablet\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"60\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"10\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"65\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"15\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"cover_gradient-gradient\\":\\"0% rgba(187, 51, 255, 0.49)|31% rgba(255, 255, 255, 0)\\",\\"cover_gradient-circle-radial\\":false,\\"cover_gradient-gradient-angle\\":\\"180\\",\\"cover_gradient-gradient-type\\":\\"linear\\",\\"cover_color-type\\":\\"cover_gradient\\",\\"background_gradient-circle-radial\\":false,\\"background_gradient-gradient-angle\\":\\"180\\",\\"background_gradient-gradient-type\\":\\"linear\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\"}},{\\"element_id\\":\\"44ew381\\",\\"cols\\":[{\\"element_id\\":\\"1mnk382\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"l7gq638\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":false,\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/about-us-image.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-left\\",\\"breakpoint_mobile\\":{\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"%\\",\\"margin_bottom\\":\\"-20\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"},\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_width_unit\\":\\"px\\",\\"width_image\\":\\"700\\"}}],\\"styling\\":{\\"padding_bottom\\":0,\\"padding_bottom_unit\\":\\"%\\",\\"padding_top\\":0,\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_opp_top\\":\\"1\\"}},{\\"element_id\\":\\"vvc0248\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"y0j8778\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>About Us<\\\\/h2>\\\\n<p>Our mission is to provide a retreat for everyone to step away from their busy lives and practice yoga within a peaceful and quiet space, while offering a variety of styles to meet the needs of everyone.<\\\\/p>\\",\\"padding_right_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_right\\":\\"20\\",\\"padding_opp_top\\":false,\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"10\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"10\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"breakpoint_tablet\\":{\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"},\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"po_left_auto\\":false,\\"po_left_unit\\":\\"px\\",\\"po_bottom_auto\\":false,\\"po_bottom_unit\\":\\"px\\",\\"po_right_auto\\":false,\\"po_right_unit\\":\\"px\\",\\"po_top_auto\\":false,\\"po_top_unit\\":\\"px\\",\\"po-type\\":\\"top\\",\\"b_sh_inset\\":false,\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"checkbox_b_ra_apply_all\\":false,\\"b_ra_left_unit\\":\\"px\\",\\"b_ra_opp_left\\":false,\\"b_ra_right_unit\\":\\"px\\",\\"b_ra_bottom_unit\\":\\"px\\",\\"b_ra_opp_bottom\\":false,\\"b_ra_top_unit\\":\\"px\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"ccg7399\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Learn more\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"button_color_bg\\":\\"purple\\",\\"icon_alignment\\":\\"left\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"%\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\",\\"padding_left\\":\\"10\\",\\"padding_right\\":\\"10\\"}}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column4-2\\",\\"styling\\":{\\"background_position\\":\\"0,100\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat-none\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth-content\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":0,\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":0,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"margin-top_opp_top\\":false,\\"margin-bottom_unit\\":\\"%\\",\\"margin-bottom\\":\\"-8\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"regular\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"margin-bottom_unit\\":\\"%\\",\\"margin-bottom\\":\\"-2\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"0\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"0\\"},\\"breakpoint_tablet\\":{\\"margin-bottom_unit\\":\\"%\\",\\"margin-bottom\\":\\"10\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"50\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"0\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"6\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"55\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"zi\\":\\"3\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}},{\\"element_id\\":\\"kefl920\\",\\"cols\\":[{\\"element_id\\":\\"bti8921\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"8z6o852\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Classes<\\\\/h2>\\\\n<p>Our certified teachers aim to provide students with access to a variety of techniques such as Vinyasa, Iyengar, Restorative meditation and many more. A variety of these techniques may be used in any given class or may be elaborated on in special-focus classes.<\\\\/p>\\\\n<p>We offer different levels of physical intensity, and different levels of experiance.<\\\\/p>\\",\\"padding_opp_left\\":false,\\"padding_opp_top\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"20\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"10\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"10\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"},\\"breakpoint_tablet\\":{\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"nl1r26\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Learn more\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"button_color_bg\\":\\"purple\\",\\"icon_alignment\\":\\"left\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"20\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\",\\"breakpoint_mobile\\":{\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"10\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"}}}],\\"styling\\":{\\"padding_bottom\\":11,\\"padding_bottom_unit\\":\\"%\\",\\"padding_top\\":11,\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_opp_top\\":\\"1\\",\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"padding_bottom\\":16,\\"padding_bottom_unit\\":\\"%\\",\\"padding_top\\":16,\\"padding_top_unit\\":\\"%\\"}}},{\\"element_id\\":\\"o8q4921\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"epba61\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":false,\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/classes-image.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-right\\",\\"width_image\\":\\"700\\"}}]}],\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column4-2\\",\\"styling\\":{\\"background_position\\":\\"100,100\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat-none\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth-content\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":0,\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":0,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"margin-bottom_unit\\":\\"%\\",\\"margin-bottom\\":\\"-6\\",\\"margin-top_opp_top\\":false,\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"0\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"regular\\"},\\"breakpoint_tablet\\":{\\"margin-bottom_unit\\":\\"%\\",\\"margin-bottom\\":\\"0\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"50\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"55\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"zi\\":\\"1\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}},{\\"element_id\\":\\"fqbz28\\",\\"cols\\":[{\\"element_id\\":\\"ijx230\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"element_id\\":\\"tcpo870\\",\\"cols\\":[{\\"element_id\\":\\"282m873\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"atfm679\\",\\"mod_settings\\":{\\"caption_image\\":\\"All Skill Levels Welcome\\",\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Strength\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-pose-a-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"i_t_r_c_opp_left\\":\\"1\\",\\"i_t_r_c_opp_bottom\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":\\"1\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}},{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"3pj3210\\",\\"mod_settings\\":{\\"caption_image\\":\\"All Skill Levels Welcome\\",\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Yin Yoga\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-pose-d-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"i_t_r_c_opp_left\\":\\"1\\",\\"i_t_r_c_opp_bottom\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":\\"1\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}}]},{\\"element_id\\":\\"k1es874\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"lduj707\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-pose-b-360x325.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"caption_image\\":\\"All Skill Levels Welcome\\",\\"title_image\\":\\"Vinyasa\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}},{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"l9kd833\\",\\"mod_settings\\":{\\"caption_image\\":\\"All Skill Levels Welcome\\",\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Iyengar\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-pose-e-360x325.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"i_t_r_c_opp_left\\":\\"1\\",\\"i_t_r_c_opp_bottom\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":\\"1\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}}]},{\\"element_id\\":\\"eyky875\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"06bg142\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-pose-c-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"caption_image\\":\\"All Skill Levels Welcome\\",\\"title_image\\":\\"Hot Yoga\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}},{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"6jhd429\\",\\"mod_settings\\":{\\"caption_image\\":\\"All Skill Levels Welcome\\",\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Restorative\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-pose-f-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"i_t_r_c_opp_left\\":\\"1\\",\\"i_t_r_c_opp_bottom\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":\\"1\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}}]}],\\"col_tablet_landscape\\":\\"column3-1\\",\\"col_tablet\\":\\"column3-1\\",\\"styling\\":{\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"40\\",\\"margin_opp_top\\":false,\\"checkbox_padding_apply_all\\":\\"1\\",\\"padding_opp_left\\":false,\\"padding_opp_top\\":false,\\"zi\\":\\"1\\"}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"w8g3688\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"More Classes\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/classes\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"button_color_bg\\":\\"purple\\",\\"icon_alignment\\":\\"left\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\",\\"alignment\\":\\"center\\"}}]}],\\"styling\\":{\\"padding_bottom_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":2,\\"padding_opp_top\\":false,\\"f_g_c_h3_h-circle-radial\\":false,\\"f_c_t_h3_h\\":\\"f_c_h3_h_solid\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"hide_anchor\\":false}},{\\"element_id\\":\\"gm21575\\",\\"cols\\":[{\\"element_id\\":\\"gc2h577\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"8h37469\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Pricing<\\\\/h2>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"40\\",\\"margin_opp_top\\":false,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#000000\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"regular\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"20\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"50\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"55\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\"}},{\\"element_id\\":\\"r48i626\\",\\"cols\\":[{\\"element_id\\":\\"431q626\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"dqjl73\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Memberships<\\\\/h3>\\\\n<p>Unlimited access for as low as $150 a month. 3 month minimum commitment. (Save over 30% on unlimited yoga classes.)<\\\\/p>\\\\n<p>$150 Monthly<\\\\/p>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"20\\",\\"margin_opp_top\\":false}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"bcrj136\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Take class\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"icon_alignment\\":\\"left\\",\\"button_color_bg\\":\\"tb_default_color\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\"}}],\\"styling\\":{\\"font_color\\":\\"#ffffff\\",\\"background_color\\":\\"#7c10f7\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"checkbox_b_ra_apply_all\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"b_ra_top\\":\\"10\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_right_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":\\"1\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"5\\",\\"padding_left\\":\\"4\\",\\"padding_bottom\\":\\"5\\",\\"padding_right\\":\\"4\\",\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"20\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"3\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"10\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"3\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"10\\"},\\"b_sh_inset\\":false,\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"b_sh_color\\":\\"#000000_0.14\\",\\"b_sh_spread\\":\\"0\\",\\"b_sh_blur\\":\\"12\\",\\"b_sh_vOffset\\":\\"3\\",\\"b_sh_hOffset\\":\\"0\\"}},{\\"element_id\\":\\"1tpm597\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"migb599\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Drop-In<\\\\/h3>\\\\n<p>Book in advance or just drop in. Any class. Ant time.<\\\\/p>\\\\n<p>$25 Drop-in classes<br \\\\/>$100 5-class package<br \\\\/>$180 10-class package<\\\\/p>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"20\\",\\"margin_opp_top\\":false}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"sc7g599\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Take class\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"icon_alignment\\":\\"left\\",\\"button_color_bg\\":\\"tb_default_color\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\"}}],\\"styling\\":{\\"font_color\\":\\"#ffffff\\",\\"background_color\\":\\"#d314b1\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"checkbox_b_ra_apply_all\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"b_ra_top\\":\\"10\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_right_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":\\"1\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"5\\",\\"padding_left\\":\\"4\\",\\"padding_bottom\\":\\"5\\",\\"padding_right\\":\\"4\\",\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"20\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"10\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"10\\"},\\"b_sh_inset\\":false,\\"b_sh_color\\":\\"#000000_0.11\\",\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_blur\\":\\"12\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_vOffset\\":\\"3\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"b_sh_hOffset\\":\\"0\\",\\"checkbox_padding_apply_all\\":false}},{\\"element_id\\":\\"6w0o931\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"93dq931\\",\\"mod_settings\\":{\\"content_text\\":\\"<h4>Addons<\\\\/h4>\\\\n<p>Unlimited access to mat and towel services every class.<\\\\/p>\\\\n<p>$20 Additional<\\\\/p>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"20\\",\\"margin_opp_top\\":false}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"g387931\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Take class\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"icon_alignment\\":\\"left\\",\\"button_color_bg\\":\\"tb_default_color\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\"}}],\\"styling\\":{\\"font_color\\":\\"#ffffff\\",\\"background_color\\":\\"#ff6000\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"checkbox_b_ra_apply_all\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"b_ra_top\\":\\"10\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_right_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":\\"1\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"5\\",\\"padding_left\\":4,\\"padding_bottom\\":\\"5\\",\\"padding_right\\":4,\\"b_sh_inset\\":false,\\"b_sh_color\\":\\"#000000_0.11\\",\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_spread\\":\\"0\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_blur\\":\\"12\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_vOffset\\":\\"3\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"b_sh_hOffset\\":\\"0\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"10\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"10\\"}}}],\\"col_tablet_landscape\\":\\"column3-1\\",\\"col_tablet\\":\\"column3-1\\"}],\\"styling\\":{\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":\\"1\\",\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"}}}],\\"styling\\":{\\"text_align\\":\\"center\\",\\"font_style_h3_regular\\":\\"normal\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"margin-top_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"10\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"10\\",\\"row_width\\":\\"fullwidth\\",\\"hide_anchor\\":false,\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"20\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"background_color\\":\\"#f6f1ff\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ffffff\\",\\"top-frame_layout\\":\\"wave4\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"7\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"15\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"5\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_height\\":\\"5\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ffffff\\",\\"top-frame_layout\\":\\"wave4\\",\\"top-frame_type\\":\\"top-presets\\"},\\"breakpoint_tablet\\":{\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"8\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_height\\":\\"8\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ffffff\\",\\"top-frame_layout\\":\\"wave4\\",\\"top-frame_type\\":\\"top-presets\\",\\"padding_bottom\\":16,\\"padding_bottom_unit\\":\\"%\\",\\"padding_top\\":16,\\"padding_top_unit\\":\\"%\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_height\\":\\"10\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ffffff\\",\\"top-frame_layout\\":\\"wave4\\",\\"top-frame_type\\":\\"top-presets\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"15\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\"},\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_size_h2\\":\\"55\\"}},{\\"element_id\\":\\"6tzy973\\",\\"cols\\":[{\\"element_id\\":\\"cv7n973\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"x3qn688\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Instructors<\\\\/h2>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"40\\",\\"margin_opp_top\\":false,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#000000\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"regular\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"50\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"55\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\"}},{\\"element_id\\":\\"b87p477\\",\\"cols\\":[{\\"element_id\\":\\"cys3478\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"28wf974\\",\\"mod_settings\\":{\\"caption_image\\":\\"Vinyasa Yoga Instructor\\",\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Maria\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/mariana-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"i_t_r_c_opp_left\\":\\"1\\",\\"i_t_r_c_opp_bottom\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":\\"1\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\",\\"i_t_m_opp_left\\":false,\\"i_t_m_opp_top\\":false}}]},{\\"element_id\\":\\"y95j479\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"3j9f975\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/julian-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"caption_image\\":\\"Iyengar Yoga Instructor\\",\\"title_image\\":\\"Juliana\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}}]},{\\"element_id\\":\\"mzvi480\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"2jk1160\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/roxane-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"caption_image\\":\\"Strength Yoga Instructor\\",\\"title_image\\":\\"Roxanne\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}}]}],\\"col_tablet_landscape\\":\\"column3-1\\",\\"col_tablet\\":\\"column3-1\\",\\"col_mobile\\":\\"column-full\\"},{\\"element_id\\":\\"hz4h342\\",\\"cols\\":[{\\"element_id\\":\\"lxg0344\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"sk5z974\\",\\"mod_settings\\":{\\"caption_image\\":\\"Hot Yoga Instructor\\",\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Rose\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/rosie-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"i_t_r_c_opp_left\\":\\"1\\",\\"i_t_r_c_opp_bottom\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":\\"1\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}}]},{\\"element_id\\":\\"o938345\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"sguk975\\",\\"mod_settings\\":{\\"caption_image\\":\\"Yin Yoga Instructor\\",\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Jeanny\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/jenny-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"i_t_r_c_opp_left\\":\\"1\\",\\"i_t_r_c_opp_bottom\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":\\"1\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}}]},{\\"element_id\\":\\"g8si345\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"pu6i976\\",\\"mod_settings\\":{\\"caption_image\\":\\"Restorative Yoga Instructor\\",\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"title_image\\":\\"Kim\\",\\"auto_fullwidth\\":\\"1\\",\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/kimberly-360x326.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"i_t_r_c_opp_left\\":\\"1\\",\\"i_t_r_c_opp_bottom\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":\\"1\\",\\"global_styles\\":\\"tb_gs415e1\\",\\"link_image\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"width_image\\":\\"360\\"}}]}],\\"col_tablet\\":\\"column3-1\\",\\"col_mobile\\":\\"column-full\\"}]}],\\"styling\\":{\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":7,\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":7,\\"f_g_c_h3_h-circle-radial\\":false,\\"f_c_t_h3_h\\":\\"f_c_h3_h_solid\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"4\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"6\\"},\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}},{\\"element_id\\":\\"vkl0885\\",\\"cols\\":[{\\"element_id\\":\\"va0m887\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"ktoy921\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Transform your body, soul, and life today<\\\\/h2>\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_size_h3\\":\\"35\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"45\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\",\\"h4_margin_bottom_unit\\":\\"px\\",\\"h4_margin_top_unit\\":\\"px\\",\\"t_shh4_blur_unit\\":\\"px\\",\\"t_shh4_vShadow_unit\\":\\"px\\",\\"t_shh4_hShadow_unit\\":\\"px\\",\\"letter_spacing_h4_unit\\":\\"px\\",\\"line_height_h4_unit\\":\\"px\\",\\"font_size_h4_unit\\":\\"px\\",\\"font_gradient_color_h4-circle-radial\\":false,\\"font_gradient_color_h4-gradient-angle\\":\\"180\\",\\"font_gradient_color_h4-gradient-type\\":\\"linear\\",\\"font_color_type_h4\\":\\"font_color_h4_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"breakpoint_tablet_landscape\\":{\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"60\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"}}}],\\"styling\\":{\\"margin-top_opp_top\\":false,\\"padding_right_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"10\\",\\"padding_opp_left\\":\\"1\\",\\"padding_right\\":\\"10\\",\\"padding_opp_top\\":false,\\"breakpoint_tablet_landscape\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"10\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"10\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"}}},{\\"element_id\\":\\"o3gs412\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"ww22745\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"zoom\\",\\"param_image\\":\\"lightbox\\",\\"auto_fullwidth\\":false,\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-image-video.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"lightbox_height_unit\\":\\"px\\",\\"lightbox_height\\":\\"500\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_width\\":\\"1200\\",\\"link_image\\":\\"https:\\\\/\\\\/www.youtube.com\\\\/watch?v=NKiGZktresA\\",\\"i_t_sh_inset\\":false,\\"checkbox_i_t_r_c_apply_all\\":\\"1\\",\\"i_t_r_c_opp_left\\":false,\\"i_t_r_c_opp_bottom\\":false,\\"b_sh_inset\\":false,\\"global_styles\\":\\"tb_gs2035e1\\",\\"width_image\\":\\"630\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"col_tablet\\":\\"column-full\\",\\"styling\\":{\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth\\",\\"background_color\\":\\"#9a4bf5\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"padding_top\\":\\"17\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":34,\\"padding_opp_top\\":false,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_h3\\":\\"#ffffff\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"700italic\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_height\\":\\"10\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ffffff\\",\\"top-frame_layout\\":\\"wave4\\",\\"top-frame_type\\":\\"top-presets\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"10\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"breakpoint_mobile\\":{\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_height\\":\\"5\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ffffff\\",\\"top-frame_layout\\":\\"wave4\\",\\"top-frame_type\\":\\"top-presets\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"17\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"5\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_color_h3\\":\\"#ffffff\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"regular\\",\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"padding_bottom\\":\\"55\\"},\\"breakpoint_tablet\\":{\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_height\\":\\"5\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ffffff\\",\\"top-frame_layout\\":\\"wave4\\",\\"top-frame_type\\":\\"top-presets\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"45\\",\\"font_color_h3\\":\\"#ffffff\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"700italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"5\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"border_inner_left_style\\":\\"solid\\",\\"border_inner_bottom_style\\":\\"solid\\",\\"border_inner_right_style\\":\\"solid\\",\\"border_inner_top_style\\":\\"solid\\",\\"border_inner-type\\":\\"top\\",\\"checkbox_padding_inner_apply_all\\":false,\\"padding_inner_left_unit\\":\\"px\\",\\"padding_inner_opp_left\\":false,\\"padding_inner_bottom_unit\\":\\"px\\",\\"padding_inner_right_unit\\":\\"px\\",\\"padding_inner_opp_top\\":false,\\"padding_inner_top_unit\\":\\"px\\",\\"background_position_inner\\":\\"50,50\\",\\"background_attachment_inner\\":\\"scroll\\",\\"background_repeat_inner\\":\\"repeat\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"50\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"17\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"50\\",\\"font_color_h3\\":\\"#ffffff\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"700italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_height\\":\\"8\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ffffff\\",\\"top-frame_layout\\":\\"wave4\\",\\"top-frame_type\\":\\"top-presets\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"39\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"17\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"6\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_color_h2\\":\\"#ffffff\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"font_size_h2\\":\\"55\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\"}},{\\"element_id\\":\\"fzyj565\\",\\"cols\\":[{\\"element_id\\":\\"wh2s567\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"fvz6892\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Locations<\\\\/h2>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":56,\\"margin_opp_top\\":false,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#000000\\",\\"font_color_type\\":\\"font_color_solid\\",\\"margin_bottom_unit\\":\\"px\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"50\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_color_h2\\":\\"#ffffff\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}}]}],\\"styling\\":{\\"padding_opp_left\\":false,\\"padding_opp_top\\":false,\\"f_g_c_h3_h-circle-radial\\":false,\\"f_c_t_h3_h\\":\\"f_c_h3_h_solid\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth\\",\\"margin-top_opp_top\\":false,\\"font_color_h3\\":\\"#ffffff\\",\\"margin-top_unit\\":\\"%\\",\\"margin-top\\":\\"-25\\",\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"%\\",\\"margin-top\\":\\"-40\\"},\\"breakpoint_tablet\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"%\\",\\"margin-top\\":\\"-40\\"},\\"breakpoint_tablet_landscape\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"%\\",\\"margin-top\\":\\"-30\\"},\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_color_h2\\":\\"#ffffff\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"font_size_h2\\":\\"55\\"}},{\\"element_id\\":\\"3s5v613\\",\\"cols\\":[{\\"element_id\\":\\"acn0615\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"element_id\\":\\"kjc1614\\",\\"cols\\":[{\\"element_id\\":\\"0mac616\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"kpaj617\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"353\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-location-a-353x309.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-top\\"}}],\\"grid_width\\":66,\\"styling\\":{\\"margin-top_opp_top\\":false}},{\\"element_id\\":\\"4w0v618\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"dk4t618\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Bloor<\\\\/h3>\\\\n<p>604-676-3941<\\\\/p>\\\\n<p>888 Bloor Street, Toronto, Canada, M4Y 1L5<\\\\/p>\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"global_styles\\":\\"tb_gs7795e5\\"}}],\\"grid_width\\":34}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column4-2\\",\\"styling\\":{\\"breakpoint_tablet\\":{\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"40\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"}}}]},{\\"element_id\\":\\"eoa7686\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"element_id\\":\\"181j749\\",\\"cols\\":[{\\"element_id\\":\\"xo1p751\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"u7qg147\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"353\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-location-b-353x309.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-top\\"}}]},{\\"element_id\\":\\"k7hy752\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"kmde829\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Queen<\\\\/h3>\\\\n<p>604-725-4731<\\\\/p>\\\\n<p>521 Queen Street, Toronto, Canada, M3N 9B4<\\\\/p>\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"global_styles\\":\\"tb_gs7795e5\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column4-2\\"}]}],\\"col_tablet\\":\\"column-full\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"margin-top\\":\\"-30\\"},\\"hide_anchor\\":false,\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_color_h3\\":\\"#e24886\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\"}},{\\"element_id\\":\\"pk87877\\",\\"cols\\":[{\\"element_id\\":\\"2rko878\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"3oak966\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":false,\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/map.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-top\\",\\"global_styles\\":\\"tb_gs2035e1\\",\\"margin_opp_left\\":false,\\"margin_opp_top\\":false,\\"width_image\\":\\"595\\"}}]},{\\"element_id\\":\\"7uaj836\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"element_id\\":\\"ntdl410\\",\\"cols\\":[{\\"element_id\\":\\"wl6i414\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"4nso974\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Contact Us<\\\\/h2>\\\\n<p>Have any questions about our classes, instructors, or memberships? Drop in or send us a message!<\\\\/p>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"40\\",\\"margin_opp_top\\":false,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#000000\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"50\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\"}},{\\"mod_name\\":\\"contact\\",\\"element_id\\":\\"r951350\\",\\"mod_settings\\":{\\"field_name_label\\":\\"Name\\",\\"field_email_label\\":\\"Email\\",\\"field_subject_label\\":\\"Subject\\",\\"field_message_label\\":\\"Message\\",\\"field_sendcopy_label\\":\\"Send a copy to myself\\",\\"field_send_label\\":\\"Send\\",\\"gdpr_label\\":\\"I consent to my submitted data being collected and stored\\",\\"field_name_require\\":\\"yes\\",\\"field_email_require\\":\\"yes\\",\\"field_name_active\\":\\"yes\\",\\"field_email_active\\":\\"yes\\",\\"field_subject_active\\":\\"yes\\",\\"field_subject_require\\":\\"yes\\",\\"field_message_active\\":\\"yes\\",\\"field_send_align\\":\\"left\\",\\"field_extra\\":\\"{ \\\\\\"fields\\\\\\": [] }\\",\\"field_order\\":\\"{}\\",\\"field_optin_label\\":\\"Subscribe to my newsletter.\\",\\"field_optin_active\\":false,\\"provider\\":\\"mailchimp\\",\\"field_sendcopy_active\\":false,\\"field_captcha_active\\":false,\\"contact_sent_from\\":\\"enable\\",\\"post_author\\":false,\\"user_role\\":\\"admin\\",\\"send_to_admins\\":false,\\"layout_contact\\":\\"style1\\"}}]}],\\"styling\\":{\\"checkbox_b_ra_apply_all\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"b_ra_top\\":\\"8\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_right_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"10\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom\\":\\"5\\",\\"padding_right\\":\\"10\\",\\"padding_opp_top\\":false,\\"padding_top\\":\\"5\\",\\"margin_left_unit\\":\\"%\\",\\"margin_top_unit\\":\\"%\\",\\"margin_left\\":\\"-10\\",\\"margin_opp_left\\":false,\\"margin_opp_top\\":false,\\"margin_top\\":\\"20\\",\\"background_color\\":\\"#ffffff\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"breakpoint_mobile\\":{\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"%\\",\\"margin_left\\":\\"0\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\",\\"margin_top\\":\\"50\\"},\\"b_sh_inset\\":false,\\"b_sh_color\\":\\"#000000_0.10\\",\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_spread\\":\\"0\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_blur\\":\\"12\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_vOffset\\":\\"3\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"b_sh_hOffset\\":\\"0\\",\\"breakpoint_tablet\\":{\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"10\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"5\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"10\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"8\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"%\\",\\"margin_left\\":\\"-15\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"%\\",\\"margin_top\\":\\"20\\"},\\"checkbox_margin_apply_all\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\"}}],\\"styling\\":{\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false}}],\\"gutter\\":\\"gutter-none\\",\\"column_h\\":1,\\"col_tablet\\":\\"column4-2\\",\\"styling\\":{\\"padding_top\\":10,\\"padding_top_unit\\":\\"%\\",\\"hide_anchor\\":false,\\"padding_opp_left\\":false,\\"padding_opp_top\\":false,\\"padding_bottom\\":\\"8\\",\\"padding_bottom_unit\\":\\"%\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}}]',
    'themify_used_global_styles' => 
    array (
      0 => 'tb_gs415e1',
      12 => 'tb_gs2035e1',
      13 => 'tb_gs7795e5',
    ),
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 165,
  'post_date' => '2020-01-10 07:08:35',
  'post_date_gmt' => '2020-01-10 07:08:35',
  'post_content' => '<!-- wp:themify-builder/canvas /--><!--themify_builder_static--><h1>Our Locations</h1> <p>Please feel free to drop by any of our locations.</p>
<h2>Our Studio Locations</h2> <p>Come visit any of our studios in the GTA</p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/classes-2-360x400.jpg" width="360" alt="classes-2" />
<h3>Bloor</h3> <p>604-676-3941</p> <p>888 Bloor Street, Toronto, Canada, M4Y 1L5</p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/02/classes-1-360x400-1-360x400.jpg" width="360" alt="classes-1-360x400" />
<h3>Christie</h3> <p>604-871- 1502</p> <p>67 Christie, Toronto, Canada, M9P 8T5</p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/classes-5-360x400.jpg" width="360" alt="classes-5" />
<h3>Spadina</h3> <p>604-357-1385</p> <p>631 Spadina Street, Toronto, Canada, M6L 1C7</p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/classes-3-360x400.jpg" width="360" alt="classes-3" />
<h3>Queen</h3> <p>604-725-4731</p> <p>521 Queen Street, Toronto, Canada, M3N 9B4</p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/classes-4-360x400.jpg" width="360" alt="classes-4" />
<h3>Bay</h3> <p>604-492-8426</p> <p>21 Bay, Toronto, Canada, M6D 2H5</p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/classes-6-360x400.jpg" width="360" alt="classes-6" />
<h3>Church</h3> <p>604-961-7541</p> <p>93 Church Street, Toronto, Canada, M3O 6B7</p><!--/themify_builder_static-->',
  'post_title' => 'Locations',
  'post_excerpt' => '',
  'post_name' => 'locations',
  'post_modified' => '2020-02-28 09:06:34',
  'post_modified_gmt' => '2020-02-28 09:06:34',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?page_id=165',
  'menu_order' => 0,
  'post_type' => 'page',
  'meta_input' => 
  array (
    'page_layout' => 'sidebar-none',
    'content_width' => 'full_width',
    'hide_page_title' => 'yes',
    'section_scrolling_mobile' => 'on',
    'mobile_menu_styles' => 'default',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"i14g127\\",\\"cols\\":[{\\"element_id\\":\\"64jv128\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"ed61561\\",\\"mod_settings\\":{\\"content_text\\":\\"<h1>Our Locations<\\\\/h1>\\\\n<p>Please feel free to drop by any of our locations.<\\\\/p>\\"}}]},{\\"element_id\\":\\"7xgw531\\",\\"grid_class\\":\\"col4-2\\"}],\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column3-2-3-1\\",\\"col_mobile\\":\\"column4-2\\",\\"styling\\":{\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/location-banner.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":\\"16\\",\\"padding_opp_top\\":false,\\"padding_top\\":\\"12\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"20\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"cover_color-type\\":\\"color\\",\\"breakpoint_mobile\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"45\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"10\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"breakpoint_tablet\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"55\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"15\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"60\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"},\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"70\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"hide_anchor\\":false,\\"background_gradient-circle-radial\\":false,\\"background_gradient-gradient-angle\\":\\"180\\",\\"background_gradient-gradient-type\\":\\"linear\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"cover_gradient-circle-radial\\":false,\\"cover_gradient-gradient-angle\\":\\"180\\",\\"cover_gradient-gradient-type\\":\\"linear\\"}},{\\"element_id\\":\\"19tk495\\",\\"cols\\":[{\\"element_id\\":\\"3g1y496\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"q03r132\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Our Studio Locations<\\\\/h2>\\\\n<p>Come visit any of our studios in the GTA<\\\\/p>\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"45\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"30\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"45\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"50\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\",\\"font_gradient_color-circle-radial\\":false,\\"font_gradient_color-gradient-angle\\":\\"180\\",\\"font_gradient_color-gradient-type\\":\\"linear\\",\\"font_color_type\\":\\"font_color_solid\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}}]}],\\"styling\\":{\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":4,\\"padding_opp_top\\":false,\\"padding_top\\":\\"4\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"background_color\\":\\"#ffffff\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"text_align\\":\\"center\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_family_h3_w\\":\\"900\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}},{\\"element_id\\":\\"t2g3716\\",\\"cols\\":[{\\"element_id\\":\\"rkq4716\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"element_id\\":\\"r76g716\\",\\"cols\\":[{\\"element_id\\":\\"plm7717\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"0n1l261\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/classes-2-360x400.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\"}}],\\"grid_width\\":66},{\\"element_id\\":\\"2evf718\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"bj69863\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Bloor<\\\\/h3>\\\\n<p>604-676-3941<\\\\/p>\\\\n<p>888 Bloor Street, Toronto, Canada, M4Y 1L5<\\\\/p>\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"global_styles\\":\\"tb_gs7795e5\\"}}],\\"grid_width\\":34}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"breakpoint_tablet\\":{\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"40\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"},\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"30\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"}},{\\"element_id\\":\\"oq04940\\",\\"cols\\":[{\\"element_id\\":\\"o9ae941\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"0cpl945\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/classes-1-360x400-1-360x400.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\"}}],\\"grid_width\\":66},{\\"element_id\\":\\"1mxq943\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"9gsj943\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Christie<\\\\/h3>\\\\n<p>604-871- 1502<\\\\/p>\\\\n<p>67 Christie, Toronto, Canada, M9P 8T5<\\\\/p>\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"global_styles\\":\\"tb_gs7795e5\\"}}],\\"grid_width\\":34}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"breakpoint_tablet\\":{\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"40\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"},\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"30\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"}},{\\"element_id\\":\\"762b101\\",\\"cols\\":[{\\"element_id\\":\\"spgj101\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"n004461\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/classes-5-360x400.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\"}}],\\"grid_width\\":66},{\\"element_id\\":\\"swlj102\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"xs89114\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Spadina<\\\\/h3>\\\\n<p>604-357-1385<\\\\/p>\\\\n<p>631 Spadina Street, Toronto, Canada, M6L 1C7<\\\\/p>\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"global_styles\\":\\"tb_gs7795e5\\"}}],\\"grid_width\\":34}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"breakpoint_tablet\\":{\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"40\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"},\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"30\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\"}}]},{\\"element_id\\":\\"u9eh719\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"element_id\\":\\"43sd719\\",\\"cols\\":[{\\"element_id\\":\\"65ds720\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"pvql717\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/classes-3-360x400.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\"}}]},{\\"element_id\\":\\"gxq5720\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"ys74194\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Queen<\\\\/h3>\\\\n<p>604-725-4731<\\\\/p>\\\\n<p>521 Queen Street, Toronto, Canada, M3N 9B4<\\\\/p>\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"global_styles\\":\\"tb_gs7795e5\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"30\\",\\"margin_opp_top\\":false}},{\\"element_id\\":\\"rsrb3\\",\\"cols\\":[{\\"element_id\\":\\"p8qa4\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"01xp363\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/classes-4-360x400.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\"}}]},{\\"element_id\\":\\"75044\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"k1lj718\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Bay<\\\\/h3>\\\\n<p>604-492-8426<\\\\/p>\\\\n<p>21 Bay, Toronto, Canada, M6D 2H5<\\\\/p>\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"global_styles\\":\\"tb_gs7795e5\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"30\\",\\"margin_opp_top\\":false}},{\\"element_id\\":\\"yqa0219\\",\\"cols\\":[{\\"element_id\\":\\"qs1j220\\",\\"grid_class\\":\\"col3-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"3q3i338\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/classes-6-360x400.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\"}}]},{\\"element_id\\":\\"vrxv220\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"ro13102\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Church<\\\\/h3>\\\\n<p>604-961-7541<\\\\/p>\\\\n<p>93 Church Street, Toronto, Canada, M3O 6B7<\\\\/p>\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"global_styles\\":\\"tb_gs7795e5\\"}}]}],\\"column_alignment\\":\\"col_align_middle\\",\\"gutter\\":\\"gutter-none\\",\\"col_tablet_landscape\\":\\"column3-2-3-1\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"30\\",\\"margin_opp_top\\":false}}]}],\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column-full\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"margin-top\\":\\"-30\\"},\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"8\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_color_h3\\":\\"#e24886\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\"}}]',
    'themify_used_global_styles' => 
    array (
      0 => 'tb_gs7795e5',
    ),
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 161,
  'post_date' => '2020-01-10 07:07:56',
  'post_date_gmt' => '2020-01-10 07:07:56',
  'post_content' => '<!-- wp:themify-builder/canvas /--><!--themify_builder_static--><h1>Pricing</h1> <p>Every dollar spent is an investment in your mental, emotional, physical health, and well-being.</p>
<h2>Find Your Best Fit</h2> <p>Beginners And Experts Always Welcome</p>
<h3>Memberships</h3> <p>Unlimited access for as low as $130 a month. 5 Month Minimum Commitment (Save over 30% on unlimited yoga classes.) </p> <p>$150 Monthly<br /><br /></p>
<a href="https://themify.me/" > Take class </a>
<h3>Drop-In</h3> <p>Book in advance or just drop in. Any class. Ant time.</p> <p>$25 Drop-in classes<br />$100 5-class package<br />$180 10-class package</p>
<a href="https://themify.me/" > Take class </a>
<h4>Addons</h4> <p>Unlimited access to Mat and Towel services every class.</p> <p>$20 additional</p>
<a href="https://themify.me/" > Take class </a>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/faq-icon.png" alt="faq-icon" />
<h2>FAQs</h2>
<ul><li><h4>Can I cancel my membership at any time?</h4><p>You can cancel your membership within 30 days of your original purchase.</p></li><li><h4>Are the memberships refundable?</h4><p>Memberships are refundable within 30 days of original purchase, with proof of purchase.</p></li><li><h4>Can I go to different locations with just one membership?</h4><p>You can visit any of our studios with one membership.</p></li><li><h4>May I suggest some yoga classes?</h4><p>We recommend our Vinyasa and Iyengar classes. These are classic forms of yoga and meditation.</p></li><li><h4>Can\'t find my answers.</h4><p>If you can not find your answers in our FAQ please send us a message and we will respond to you as fast as possible.</p></li></ul><!--/themify_builder_static-->',
  'post_title' => 'Pricing',
  'post_excerpt' => '',
  'post_name' => 'pricing',
  'post_modified' => '2020-02-28 09:05:37',
  'post_modified_gmt' => '2020-02-28 09:05:37',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?page_id=161',
  'menu_order' => 0,
  'post_type' => 'page',
  'meta_input' => 
  array (
    'page_layout' => 'sidebar-none',
    'content_width' => 'full_width',
    'hide_page_title' => 'yes',
    'section_scrolling_mobile' => 'on',
    'mobile_menu_styles' => 'default',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"hj9d870\\",\\"cols\\":[{\\"element_id\\":\\"g830870\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"6xkz871\\",\\"mod_settings\\":{\\"content_text\\":\\"<h1>Pricing<\\\\/h1>\\\\n<p>Every dollar spent is an investment in your mental, emotional, physical health, and well-being.<\\\\/p>\\",\\"breakpoint_mobile\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"},\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":false,\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":false,\\"margin_top_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"}}]},{\\"element_id\\":\\"ypyi871\\",\\"grid_class\\":\\"col4-2\\"}],\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":\\"16\\",\\"padding_opp_top\\":false,\\"padding_top\\":\\"12\\",\\"background_position\\":\\"50,46.25\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/pricing-hero-bg.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"20\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"font_style_h3_regular\\":\\"normal\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"regular\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"65\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"breakpoint_mobile\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"45\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"30\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"20\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"8\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"breakpoint_tablet\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"55\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"10\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"55\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"15\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"background_gradient-circle-radial\\":false,\\"background_gradient-gradient-angle\\":\\"180\\",\\"background_gradient-gradient-type\\":\\"linear\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"background_color\\":\\"#f0e9f9\\"}},{\\"element_id\\":\\"dypl482\\",\\"cols\\":[{\\"element_id\\":\\"mmtg483\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"orlo255\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Find Your Best Fit<\\\\/h2>\\\\n<p>Beginners And Experts Always Welcome<\\\\/p>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"60\\",\\"margin_opp_top\\":false,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#000000\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"font_size_h3\\":\\"30\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"text_align\\":\\"center\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\",\\"font_color\\":\\"#000000\\",\\"font_gradient_color-circle-radial\\":false,\\"font_gradient_color-gradient-angle\\":\\"180\\",\\"font_gradient_color-gradient-type\\":\\"linear\\",\\"font_color_type\\":\\"font_color_solid\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"50\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"margin_bottom_unit\\":\\"px\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_top_unit\\":\\"px\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\"}},{\\"element_id\\":\\"7zoj484\\",\\"cols\\":[{\\"element_id\\":\\"knzn485\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"lgyn485\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Memberships<\\\\/h3>\\\\n<p>Unlimited access for as low as $130 a month. 5 Month Minimum Commitment (Save over 30% on unlimited yoga classes.) <\\\\/p>\\\\n<p>$150 Monthly<br \\\\/><br \\\\/><\\\\/p>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"20\\",\\"margin_opp_top\\":false}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"gqmo485\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Take class\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"icon_alignment\\":\\"left\\",\\"button_color_bg\\":\\"tb_default_color\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\"}}],\\"styling\\":{\\"font_color\\":\\"#ffffff\\",\\"background_color\\":\\"#7c10f7\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"checkbox_b_ra_apply_all\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"b_ra_top\\":\\"10\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_right_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":\\"1\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"5\\",\\"padding_left\\":\\"4\\",\\"padding_bottom\\":\\"5\\",\\"padding_right\\":\\"4\\",\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"20\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"3\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"10\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"3\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"10\\"},\\"b_sh_inset\\":false,\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"b_sh_color\\":\\"#000000_0.14\\",\\"b_sh_spread\\":\\"0\\",\\"b_sh_blur\\":\\"12\\",\\"b_sh_vOffset\\":\\"3\\",\\"b_sh_hOffset\\":\\"0\\"}},{\\"element_id\\":\\"umea485\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"4c25486\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Drop-In<\\\\/h3>\\\\n<p>Book in advance or just drop in. Any class. Ant time.<\\\\/p>\\\\n<p>$25 Drop-in classes<br \\\\/>$100 5-class package<br \\\\/>$180 10-class package<\\\\/p>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"20\\",\\"margin_opp_top\\":false}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"ybmn486\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Take class\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"icon_alignment\\":\\"left\\",\\"button_color_bg\\":\\"tb_default_color\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\"}}],\\"styling\\":{\\"font_color\\":\\"#ffffff\\",\\"background_color\\":\\"#d314b1\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"checkbox_b_ra_apply_all\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"b_ra_top\\":\\"10\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_right_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":\\"1\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"5\\",\\"padding_left\\":\\"4\\",\\"padding_bottom\\":\\"5\\",\\"padding_right\\":\\"4\\",\\"breakpoint_mobile\\":{\\"margin-bottom_unit\\":\\"px\\",\\"margin-bottom\\":\\"20\\",\\"margin-top_opp_top\\":false,\\"margin-top_unit\\":\\"px\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"10\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"10\\"},\\"b_sh_inset\\":false,\\"b_sh_color\\":\\"#000000_0.11\\",\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_spread\\":\\"0\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_blur\\":\\"12\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_vOffset\\":\\"3\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"b_sh_hOffset\\":\\"0\\",\\"checkbox_padding_apply_all\\":false}},{\\"element_id\\":\\"f35n486\\",\\"grid_class\\":\\"col3-1\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"bqgw486\\",\\"mod_settings\\":{\\"content_text\\":\\"<h4>Addons<\\\\/h4>\\\\n<p>Unlimited access to Mat and Towel services every class.<\\\\/p>\\\\n<p>$20 additional<\\\\/p>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"20\\",\\"margin_opp_top\\":false}},{\\"mod_name\\":\\"buttons\\",\\"element_id\\":\\"4mdf486\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Take class\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"icon_alignment\\":\\"left\\",\\"button_color_bg\\":\\"tb_default_color\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"rounded\\",\\"buttons_size\\":\\"normal\\"}}],\\"styling\\":{\\"font_color\\":\\"#ffffff\\",\\"background_color\\":\\"#ff6000\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"checkbox_b_ra_apply_all\\":\\"1\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"b_ra_top\\":\\"10\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_right_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":\\"1\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":\\"5\\",\\"padding_left\\":\\"4\\",\\"padding_bottom\\":\\"5\\",\\"padding_right\\":\\"4\\",\\"b_sh_inset\\":false,\\"b_sh_color\\":\\"#000000_0.11\\",\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_spread\\":\\"0\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_blur\\":\\"12\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_vOffset\\":\\"3\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"b_sh_hOffset\\":\\"0\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"%\\",\\"padding_left\\":\\"5\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"10\\",\\"padding_right_unit\\":\\"%\\",\\"padding_right\\":\\"5\\",\\"padding_opp_top\\":\\"1\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"10\\"},\\"checkbox_padding_apply_all\\":false}}],\\"col_tablet_landscape\\":\\"column3-1\\",\\"col_tablet\\":\\"column3-1\\",\\"col_mobile\\":\\"column-full\\"}]}],\\"styling\\":{\\"text_align\\":\\"center\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"margin-top_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_opp_top\\":\\"1\\",\\"padding_top\\":5,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":5,\\"row_width\\":\\"fullwidth\\",\\"hide_anchor\\":false,\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_type\\":\\"top-presets\\",\\"background_position\\":\\"50,50\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"repeat\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"7\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"9\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"5\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_height\\":\\"5\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ffffff\\",\\"top-frame_layout\\":\\"wave4\\",\\"top-frame_type\\":\\"top-presets\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\"},\\"breakpoint_tablet\\":{\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"8\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_height\\":\\"8\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ffffff\\",\\"top-frame_layout\\":\\"wave4\\",\\"top-frame_type\\":\\"top-presets\\",\\"padding_bottom\\":\\"16\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top\\":\\"10\\",\\"padding_top_unit\\":\\"%\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_height\\":\\"10\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_width\\":\\"100\\",\\"top-frame_color\\":\\"#ffffff\\",\\"top-frame_layout\\":\\"wave4\\",\\"top-frame_type\\":\\"top-presets\\",\\"bottom-frame_location\\":\\"in_bellow\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"15\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\"},\\"background_gradient-circle-radial\\":false,\\"background_gradient-gradient-angle\\":\\"180\\",\\"background_gradient-gradient-type\\":\\"linear\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}},{\\"element_id\\":\\"s70j762\\",\\"cols\\":[{\\"element_id\\":\\"dru8762\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"4h51324\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":false,\\"appearance_image\\":false,\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/faq-icon.png\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-center\\"}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"jbmv483\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>FAQs<\\\\/h2>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":50,\\"margin_opp_top\\":false,\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"60\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"text_align\\":\\"center\\",\\"font_color\\":\\"#000000\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"35\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"regular\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"55\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"margin_bottom_unit\\":\\"px\\",\\"checkbox_margin_apply_all\\":false,\\"margin_left_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_top_unit\\":\\"px\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}},{\\"mod_name\\":\\"accordion\\",\\"element_id\\":\\"sxhs221\\",\\"mod_settings\\":{\\"content_accordion\\":[{\\"title_accordion\\":\\"Can I cancel my membership at any time?\\",\\"text_accordion\\":\\"<p>You can cancel your membership within 30 days of your original purchase.<\\\\/p>\\",\\"default_accordion\\":\\"open\\"},{\\"title_accordion\\":\\"Are the memberships refundable?\\",\\"text_accordion\\":\\"<p>Memberships are refundable within 30 days of original purchase, with proof of purchase.<\\\\/p>\\",\\"default_accordion\\":\\"closed\\"},{\\"title_accordion\\":\\"Can I go to different locations with just one membership?\\",\\"text_accordion\\":\\"<p>You can visit any of our studios with one membership.<\\\\/p>\\",\\"default_accordion\\":\\"closed\\"},{\\"title_accordion\\":\\"May I suggest some yoga classes?\\",\\"text_accordion\\":\\"<p>We recommend our Vinyasa and Iyengar classes. These are classic forms of yoga and meditation.<\\\\/p>\\",\\"default_accordion\\":\\"closed\\"},{\\"title_accordion\\":\\"Can\\\'t find my answers.\\",\\"text_accordion\\":\\"<p>If you can not find your answers in our FAQ please send us a message and we will respond to you as fast as possible.<\\\\/p>\\",\\"default_accordion\\":\\"closed\\"}],\\"accordion_appearance_accordion\\":false,\\"expand_collapse_accordion\\":\\"toggle\\",\\"icon_active_accordion\\":\\"ti-minus\\",\\"icon_accordion\\":\\"ti-plus\\",\\"color_accordion\\":\\"transparent\\",\\"t_sh_t_blur_unit\\":\\"px\\",\\"t_sh_t_vShadow_unit\\":\\"px\\",\\"t_sh_t_hShadow_unit\\":\\"px\\",\\"f_s_t_regular\\":\\"normal\\",\\"l_s_t_unit\\":\\"px\\",\\"line_height_title_unit\\":\\"px\\",\\"font_size_title_unit\\":\\"px\\",\\"font_size_title\\":\\"22\\",\\"font_color_title\\":\\"#000000\\",\\"font_family_title_w\\":\\"600\\",\\"font_family_title\\":\\"Poppins\\",\\"b_ct_left_style\\":\\"solid\\",\\"b_ct_bottom_style\\":\\"none\\",\\"b_ct_right_style\\":\\"solid\\",\\"b_ct_top_style\\":\\"solid\\",\\"b_ct_top_width\\":\\"1\\",\\"b_ct_top_color\\":\\"#e0e0e0\\",\\"b_ct-type\\":\\"bottom\\",\\"b_ct_bottom_width\\":\\"1\\",\\"b_ct_bottom_color\\":\\"#e0e0e0\\",\\"checkbox_p_ct_apply_all\\":false,\\"p_ct_left_unit\\":\\"px\\",\\"p_ct_opp_left\\":false,\\"p_ct_bottom_unit\\":\\"px\\",\\"p_ct_right_unit\\":\\"px\\",\\"p_ct_opp_top\\":false,\\"p_ct_top_unit\\":\\"px\\",\\"p_ct_top\\":\\"18\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\",\\"l_s_t\\":\\"1\\",\\"checkbox_r_c_ct_apply_all\\":false,\\"r_c_ct_left_unit\\":\\"px\\",\\"r_c_ct_opp_left\\":false,\\"r_c_ct_right_unit\\":\\"px\\",\\"r_c_ct_bottom_unit\\":\\"px\\",\\"r_c_ct_opp_bottom\\":false,\\"r_c_ct_top_unit\\":\\"px\\",\\"icon_size_unit\\":\\"px\\",\\"icon_active_color\\":\\"#000000\\",\\"icon_color\\":\\"#000000\\",\\"checkbox_m_a_i_apply_all\\":false,\\"m_a_i_left_unit\\":\\"px\\",\\"m_a_i_opp_left\\":false,\\"m_a_i_bottom_unit\\":\\"px\\",\\"m_a_i_right_unit\\":\\"px\\",\\"m_a_i_right\\":\\"20\\",\\"m_a_i_opp_top\\":false,\\"m_a_i_top_unit\\":\\"px\\",\\"breakpoint_mobile\\":{\\"t_sh_t_blur_unit\\":\\"px\\",\\"t_sh_t_vShadow_unit\\":\\"px\\",\\"t_sh_t_hShadow_unit\\":\\"px\\",\\"f_s_t_regular\\":\\"normal\\",\\"l_s_t_unit\\":\\"px\\",\\"l_s_t\\":\\"1\\",\\"line_height_title_unit\\":\\"px\\",\\"font_size_title_unit\\":\\"px\\",\\"font_size_title\\":\\"16\\",\\"font_color_title\\":\\"#000000\\",\\"font_family_title_w\\":\\"600\\",\\"font_family_title\\":\\"Poppins\\"},\\"breakpoint_tablet\\":{\\"t_sh_t_blur_unit\\":\\"px\\",\\"t_sh_t_vShadow_unit\\":\\"px\\",\\"t_sh_t_hShadow_unit\\":\\"px\\",\\"f_s_t_regular\\":\\"normal\\",\\"l_s_t_unit\\":\\"px\\",\\"l_s_t\\":\\"1\\",\\"line_height_title_unit\\":\\"px\\",\\"font_size_title_unit\\":\\"px\\",\\"font_size_title\\":\\"20\\",\\"font_color_title\\":\\"#000000\\",\\"font_family_title_w\\":\\"600\\",\\"font_family_title\\":\\"Poppins\\"},\\"breakpoint_tablet_landscape\\":{\\"t_sh_t_blur_unit\\":\\"px\\",\\"t_sh_t_vShadow_unit\\":\\"px\\",\\"t_sh_t_hShadow_unit\\":\\"px\\",\\"f_s_t_regular\\":\\"normal\\",\\"l_s_t_unit\\":\\"px\\",\\"l_s_t\\":\\"1\\",\\"line_height_title_unit\\":\\"px\\",\\"font_size_title_unit\\":\\"px\\",\\"font_size_title\\":\\"20\\",\\"font_color_title\\":\\"#000000\\",\\"font_family_title_w\\":\\"600\\",\\"font_family_title\\":\\"Poppins\\"}}}]}],\\"styling\\":{\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":\\"5\\",\\"padding_opp_top\\":false,\\"padding_top\\":\\"2\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}}]',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 163,
  'post_date' => '2020-01-10 07:08:18',
  'post_date_gmt' => '2020-01-10 07:08:18',
  'post_content' => '<!-- wp:themify-builder/canvas /--><!--themify_builder_static--><h1>Teacher Training</h1> <p>We share our passion for yoga with everyone, any shape, size, age, and status. Anyone can do it.</p>
<h2>Learn how to be a yoga teacher</h2> <p>We have created a program that will get you ready to teach creative, and transformative yoga classes. Our training is led by professionals who will prepare you to work in a yoga studio and traditional fitness.</p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/teacher-training-a-500x280.jpg" width="500" alt="teacher-training-a" />
<h3><a href="https://themify.me/">200-Hour Cross-Disciplinary Training</a></h3> <p>Our training is a 200-hour course with our world class teachers. You will get exposure to the most important ideas to inspire your teaching and practices.</p> <p><a href="https://themify.me/">Read more</a></p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-teacher-3-500x280.jpg" width="500" alt="yoga-teacher-3" />
<h3><a href="https://themify.me/">Experienced Faculty</a></h3> <p>Our experienced and passionate faculty draw on diverse yoga traditions and actively teach in the yoga community.</p> <p><a href="https://themify.me/">Read more</a></p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-gallery-5-500x280.jpg" width="500" height="280" alt="yoga-gallery-5" />
<h3><a href="https://themify.me/">Unparalleled Support</a></h3> <p>Our training experience has the option for personal development coaching that will help develop your talents and tap into your potential, as well as facilitate your employability. </p> <p><a href="https://themify.me/">Read more</a></p>
<img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-teacher-2-500x280.jpg" width="500" height="280" alt="yoga-teacher-2" />
<h3><a href="https://themify.me/">200-Hour Study Program</a></h3> <p>This comprehensive course combines practice application and study. Students will have the opportunity to learn the material through a hands on approach.</p> <p><a href="https://themify.me/">Read more</a></p><!--/themify_builder_static-->',
  'post_title' => 'Teacher',
  'post_excerpt' => '',
  'post_name' => 'teacher',
  'post_modified' => '2020-02-28 09:06:09',
  'post_modified_gmt' => '2020-02-28 09:06:09',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?page_id=163',
  'menu_order' => 0,
  'post_type' => 'page',
  'meta_input' => 
  array (
    'page_layout' => 'sidebar-none',
    'content_width' => 'full_width',
    'hide_page_title' => 'yes',
    'section_scrolling_mobile' => 'on',
    'mobile_menu_styles' => 'default',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"r7l4361\\",\\"cols\\":[{\\"element_id\\":\\"tbn0362\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"8lbx362\\",\\"mod_settings\\":{\\"content_text\\":\\"<h1>Teacher Training<\\\\/h1>\\\\n<p>We share our passion for yoga with everyone, any shape, size, age, and status. Anyone can do it.<\\\\/p>\\",\\"breakpoint_mobile\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\"}}}]},{\\"element_id\\":\\"jt3m363\\",\\"grid_class\\":\\"col4-2\\"}],\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"hide_anchor\\":false,\\"row_width\\":\\"fullwidth\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":\\"16\\",\\"padding_opp_top\\":false,\\"padding_top\\":\\"12\\",\\"background_position\\":\\"50,46.25\\",\\"background_zoom\\":false,\\"background_attachment\\":\\"scroll\\",\\"background_repeat\\":\\"fullcover\\",\\"background_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/classes-bg.jpg\\",\\"background_video_options\\":\\"mute\\",\\"background_slider_speed\\":\\"2000\\",\\"background_slider_mode\\":\\"fullcover\\",\\"background_slider_size\\":\\"large\\",\\"background_type\\":\\"image\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"20\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_type\\":\\"top-presets\\",\\"font_style_h3_regular\\":\\"normal\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"regular\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"65\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"breakpoint_mobile\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"45\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"30\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"20\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"8\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"breakpoint_tablet\\":{\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"50\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"10\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"font_style_h1_regular\\":\\"italic\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_size_h1_unit\\":\\"px\\",\\"font_size_h1\\":\\"65\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"font_color_type_h1\\":\\"font_color_h1_solid\\",\\"font_family_h1_w\\":\\"900italic\\",\\"font_family_h1\\":\\"Playfair Display\\",\\"bottom-frame_location\\":\\"in_front\\",\\"bottom-frame_height_unit\\":\\"%\\",\\"bottom-frame_height\\":\\"15\\",\\"bottom-frame_width_unit\\":\\"%\\",\\"bottom-frame_width\\":\\"100\\",\\"bottom-frame_color\\":\\"#ffffff\\",\\"bottom-frame_layout\\":\\"wave4\\",\\"bottom-frame_type\\":\\"bottom-presets\\",\\"top-frame_location\\":\\"in_bellow\\",\\"top-frame_height_unit\\":\\"%\\",\\"top-frame_width_unit\\":\\"%\\",\\"top-frame_layout\\":\\"none\\",\\"top-frame_type\\":\\"top-presets\\"},\\"h1_margin_bottom_unit\\":\\"px\\",\\"h1_margin_top_unit\\":\\"px\\",\\"t_shh1_blur_unit\\":\\"px\\",\\"t_shh1_vShadow_unit\\":\\"px\\",\\"t_shh1_hShadow_unit\\":\\"px\\",\\"letter_spacing_h1_unit\\":\\"px\\",\\"line_height_h1_unit\\":\\"px\\",\\"font_gradient_color_h1-circle-radial\\":false,\\"font_gradient_color_h1-gradient-angle\\":\\"180\\",\\"font_gradient_color_h1-gradient-type\\":\\"linear\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"background_gradient-circle-radial\\":false,\\"background_gradient-gradient-angle\\":\\"180\\",\\"background_gradient-gradient-type\\":\\"linear\\",\\"cover_gradient-gradient\\":\\"0% rgba(255, 255, 255, 0.45)|100% rgba(255, 255, 255, 0)\\",\\"cover_gradient-circle-radial\\":false,\\"cover_gradient-gradient-angle\\":\\"99\\",\\"cover_gradient-gradient-type\\":\\"linear\\",\\"cover_color-type\\":\\"cover_gradient\\"}},{\\"element_id\\":\\"svc5756\\",\\"cols\\":[{\\"element_id\\":\\"sfnp757\\",\\"grid_class\\":\\"col-full\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"7z9a160\\",\\"mod_settings\\":{\\"content_text\\":\\"<h2>Learn how to be a yoga teacher<\\\\/h2>\\\\n<p>We have created a program that will get you ready to teach creative, and transformative yoga classes. Our training is led by professionals who will prepare you to work in a yoga studio and traditional fitness.<\\\\/p>\\",\\"margin_opp_left\\":false,\\"margin_bottom\\":\\"40\\",\\"margin_opp_top\\":false,\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_color\\":\\"#000000\\",\\"font_color_type\\":\\"font_color_solid\\",\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"normal\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"30\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"breakpoint_tablet\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"40\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\"},\\"breakpoint_tablet_landscape\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"font_style_h3_regular\\":\\"italic\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"50\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"font_family_h3_w\\":\\"900italic\\",\\"font_family_h3\\":\\"Playfair Display\\",\\"checkbox_padding_apply_all\\":false,\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":false,\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":false,\\"padding_top_unit\\":\\"px\\"},\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\",\\"font_gradient_color-circle-radial\\":false,\\"font_gradient_color-gradient-angle\\":\\"180\\",\\"font_gradient_color-gradient-type\\":\\"linear\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"text_align\\":\\"center\\",\\"font_size_h3_unit\\":\\"px\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}}]}],\\"styling\\":{\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"padding_top\\":\\"5\\",\\"padding_top_unit\\":\\"%\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":\\"2\\",\\"padding_opp_top\\":false,\\"breakpoint_mobile\\":{\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\"},\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"50\\",\\"font_gradient_color_h3-circle-radial\\":false,\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_family_h3_w\\":\\"900italic\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\",\\"h2_margin_bottom_unit\\":\\"px\\",\\"h2_margin_top_unit\\":\\"px\\",\\"t_shh2_blur_unit\\":\\"px\\",\\"t_shh2_vShadow_unit\\":\\"px\\",\\"t_shh2_hShadow_unit\\":\\"px\\",\\"letter_spacing_h2_unit\\":\\"px\\",\\"line_height_h2_unit\\":\\"px\\",\\"font_size_h2_unit\\":\\"px\\",\\"font_size_h2\\":\\"55\\",\\"font_gradient_color_h2-circle-radial\\":false,\\"font_gradient_color_h2-gradient-angle\\":\\"180\\",\\"font_gradient_color_h2-gradient-type\\":\\"linear\\",\\"font_color_type_h2\\":\\"font_color_h2_solid\\"}},{\\"element_id\\":\\"ruh0974\\",\\"cols\\":[{\\"element_id\\":\\"6rba976\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"jacv671\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":false,\\"width_image\\":\\"500\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/teacher-training-a-500x280.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-top\\",\\"global_styles\\":\\"tb_gs3455e2\\"}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"4ml9244\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3><a href=\\\\\\"https:\\\\/\\\\/themify.me\\\\/\\\\\\">200-Hour Cross-Disciplinary Training<\\\\/a><\\\\/h3>\\\\n<p>Our training is a 200-hour course with our world class teachers. You will get exposure to the most important ideas to inspire your teaching and practices.<\\\\/p>\\\\n<p><a href=\\\\\\"https:\\\\/\\\\/themify.me\\\\/\\\\\\">Read more<\\\\/a><\\\\/p>\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"global_styles\\":\\"tb_gs3475e2\\"}},{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"vqer146\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":false,\\"width_image\\":\\"500\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-teacher-3-500x280.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-top\\",\\"global_styles\\":\\"tb_gs3455e2\\"}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"g08l840\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3><a href=\\\\\\"https:\\\\/\\\\/themify.me\\\\/\\\\\\">Experienced Faculty<\\\\/a><\\\\/h3>\\\\n<p>Our experienced and passionate faculty draw on diverse yoga traditions and actively teach in the yoga community.<\\\\/p>\\\\n<p><a href=\\\\\\"https:\\\\/\\\\/themify.me\\\\/\\\\\\">Read more<\\\\/a><\\\\/p>\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"global_styles\\":\\"tb_gs3475e2\\"}}]},{\\"element_id\\":\\"vq9c587\\",\\"grid_class\\":\\"col4-2\\",\\"modules\\":[{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"ynux793\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":false,\\"width_image\\":\\"500\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-gallery-5-500x280.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-top\\",\\"global_styles\\":\\"tb_gs3455e2\\",\\"height_image\\":\\"280\\"}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"pzwx524\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3><a href=\\\\\\"https:\\\\/\\\\/themify.me\\\\/\\\\\\">Unparalleled Support<\\\\/a><\\\\/h3>\\\\n<p>Our training experience has the option for personal development coaching that will help develop your talents and tap into your potential, as well as facilitate your employability. <\\\\/p>\\\\n<p><a href=\\\\\\"https:\\\\/\\\\/themify.me\\\\/\\\\\\">Read more<\\\\/a><\\\\/p>\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"global_styles\\":\\"tb_gs3475e2\\"}},{\\"mod_name\\":\\"image\\",\\"element_id\\":\\"a65p23\\",\\"mod_settings\\":{\\"image_zoom_icon\\":false,\\"param_image\\":\\"regular\\",\\"height_image\\":\\"280\\",\\"auto_fullwidth\\":false,\\"width_image\\":\\"500\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-teacher-2-500x280.jpg\\",\\"caption_on_overlay\\":false,\\"style_image\\":\\"image-top\\",\\"global_styles\\":\\"tb_gs3455e2\\"}},{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"077u720\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3><a href=\\\\\\"https:\\\\/\\\\/themify.me\\\\/\\\\\\">200-Hour Study Program<\\\\/a><\\\\/h3>\\\\n<p>This comprehensive course combines practice application and study. Students will have the opportunity to learn the material through a hands on approach.<\\\\/p>\\\\n<p><a href=\\\\\\"https:\\\\/\\\\/themify.me\\\\/\\\\\\">Read more<\\\\/a><\\\\/p>\\",\\"b_ra_opp_left\\":false,\\"b_ra_opp_bottom\\":false,\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"global_styles\\":\\"tb_gs3475e2\\"}}]}],\\"col_tablet_landscape\\":\\"column4-2\\",\\"col_tablet\\":\\"column4-2\\",\\"col_mobile\\":\\"column-full\\",\\"styling\\":{\\"padding_bottom_unit\\":\\"%\\",\\"padding_opp_left\\":false,\\"padding_bottom\\":\\"5\\",\\"padding_opp_top\\":false}}]',
    'themify_used_global_styles' => 
    array (
      0 => 'tb_gs3455e2',
      1 => 'tb_gs3475e2',
    ),
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 779,
  'post_date' => '2020-02-28 07:52:04',
  'post_date_gmt' => '2020-02-28 07:52:04',
  'post_content' => '<!--themify_builder_static--><h3>Iyengar Yoga</h3> <p>All Levels Welcome Beginner Friendly</p><!--/themify_builder_static-->',
  'post_title' => 'Rounded Text Box',
  'post_excerpt' => '',
  'post_name' => 'tb_gs7795e5',
  'post_modified' => '2020-02-28 08:47:48',
  'post_modified_gmt' => '2020-02-28 08:47:48',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/tglobal-style/rounded-text-box/',
  'menu_order' => 0,
  'post_type' => 'tglobal_style',
  'meta_input' => 
  array (
    'themify_global_style_type' => 'text',
    'hide_page_title' => 'yes',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
    '_wp_old_slug' => 'rounded-text-box',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"row5e58c6a423c68\\",\\"cols\\":[{\\"element_id\\":\\"col5e58c6a423c68\\",\\"modules\\":[{\\"mod_name\\":\\"text\\",\\"element_id\\":\\"mod5e58c6a423c68\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Iyengar Yoga<\\\\/h3>\\\\n<p>All Levels Welcome Beginner Friendly<\\\\/p>\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":\\"1\\",\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":\\"false\\",\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":\\"false\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"10\\",\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"em\\",\\"margin_left\\":\\"0\\",\\"margin_opp_left\\":\\"false\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\",\\"margin_top\\":\\"30\\"},\\"checkbox_b_ra_apply_all\\":\\"1\\",\\"b_ra_left_unit\\":\\"px\\",\\"b_ra_opp_left\\":\\"false\\",\\"b_ra_right_unit\\":\\"px\\",\\"b_ra_bottom_unit\\":\\"px\\",\\"b_ra_opp_bottom\\":\\"false\\",\\"b_ra_top_unit\\":\\"px\\",\\"b_ra_top\\":\\"8\\",\\"checkbox_padding_apply_all\\":\\"1\\",\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":\\"false\\",\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":\\"false\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"10\\",\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"background_color\\":\\"#ffffff\\",\\"background_image-circle-radial\\":\\"false\\",\\"background_image-gradient-angle\\":\\"180\\",\\"background_image-gradient-type\\":\\"linear\\",\\"background_image-type\\":\\"image\\",\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"%\\",\\"margin_left\\":\\"-25\\",\\"margin_opp_left\\":\\"false\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\",\\"b_sh_inset\\":\\"false\\",\\"b_sh_color\\":\\"#000000_0.09\\",\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_spread\\":\\"0\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_blur\\":\\"12\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_vOffset\\":\\"3\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"b_sh_hOffset\\":\\"0\\"}}]}]}]',
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 727,
  'post_date' => '2020-02-28 00:46:27',
  'post_date_gmt' => '2020-02-28 00:46:27',
  'post_content' => '<!--themify_builder_static--><img src="https://themify.me/demo/themes/ultra-yoga/files/2020/02/asian-doing-yoga-360x389.jpg" width="360" alt="asian-doing-yoga" srcset="https://themify.me/demo/themes/ultra-yoga/files/2020/02/asian-doing-yoga-360x389.jpg 360w, https://themify.me/demo/themes/ultra-yoga/files/2020/02/asian-doing-yoga.jpg 400w" sizes="(max-width: 360px) 100vw, 360px" /><!--/themify_builder_static-->',
  'post_title' => 'Class Image Shadow',
  'post_excerpt' => '',
  'post_name' => 'tb_gs7275e5',
  'post_modified' => '2020-02-28 08:31:23',
  'post_modified_gmt' => '2020-02-28 08:31:23',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/tglobal-style/class-image-shadow/',
  'menu_order' => 0,
  'post_type' => 'tglobal_style',
  'meta_input' => 
  array (
    'themify_global_style_type' => 'image',
    'hide_page_title' => 'yes',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
    '_wp_old_slug' => 'class-image-shadow',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"row5e5862e3aa810\\",\\"cols\\":[{\\"element_id\\":\\"col5e5862e3aa810\\",\\"modules\\":[{\\"element_id\\":\\"mod5e5862e3aa810\\",\\"mod_name\\":\\"image\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"false\\",\\"param_image\\":\\"regular\\",\\"auto_fullwidth\\":\\"false\\",\\"width_image\\":\\"360\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/02\\\\/asian-doing-yoga-360x389.jpg\\",\\"caption_on_overlay\\":\\"false\\",\\"style_image\\":\\"image-center\\",\\"breakpoint_mobile\\":{\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":\\"false\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\",\\"checkbox_i_t_m_apply_all\\":\\"false\\",\\"i_t_m_left_unit\\":\\"px\\",\\"i_t_m_opp_left\\":\\"false\\",\\"i_t_m_bottom_unit\\":\\"px\\",\\"i_t_m_right_unit\\":\\"px\\",\\"i_t_m_opp_top\\":\\"false\\",\\"i_t_m_top_unit\\":\\"px\\"}}}]}]}]',
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 347,
  'post_date' => '2020-01-27 07:48:13',
  'post_date_gmt' => '2020-01-27 07:48:13',
  'post_content' => '<!--themify_builder_static--><h3><a href=\\"https://themify.me/\\">200-Hour Yoga Teacher Training</a></h3> <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Proin congue, est at egestas facilisis, lectus est dapibus leo, quis pellentesque libero risus sit amet justo. Etiam sollicitudin ante quam, vitae hendrerit elit tincidunt quis. Suspendisse pulvinar odio in lectus blandit finibus. Proin convallis neque</p> <p><a href=\\"https://themify.me/\\">Read more</a></p><!--/themify_builder_static-->',
  'post_title' => 'Teacher Training Text',
  'post_excerpt' => '',
  'post_name' => 'tb_gs3475e2',
  'post_modified' => '2020-02-10 22:44:37',
  'post_modified_gmt' => '2020-02-10 22:44:37',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/tglobal-style/teacher-training-text/',
  'menu_order' => 0,
  'post_type' => 'tglobal_style',
  'meta_input' => 
  array (
    'themify_global_style_type' => 'text',
    'hide_page_title' => 'yes',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
    '_wp_old_slug' => 'teacher-training-text',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"row5e2e95bd052d5\\",\\"cols\\":[{\\"element_id\\":\\"col5e2e95bd052d5\\",\\"modules\\":[{\\"element_id\\":\\"mod5e2e95bd052d5\\",\\"mod_name\\":\\"text\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3><a href=\\\\\\\\\\\\\\"https:\\\\/\\\\/themify.me\\\\/\\\\\\\\\\\\\\">200-Hour Yoga Teacher Training<\\\\/a><\\\\/h3>\\\\n<p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Proin congue, est at egestas facilisis, lectus est dapibus leo, quis pellentesque libero risus sit amet justo. Etiam sollicitudin ante quam, vitae hendrerit elit tincidunt quis. Suspendisse pulvinar odio in lectus blandit finibus. Proin convallis neque<\\\\/p>\\\\n<p><a href=\\\\\\\\\\\\\\"https:\\\\/\\\\/themify.me\\\\/\\\\\\\\\\\\\\">Read more<\\\\/a><\\\\/p>\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_right_unit\\":\\"%\\",\\"padding_left_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_left\\":\\"6\\",\\"padding_opp_left\\":\\"1\\",\\"padding_bottom\\":\\"4\\",\\"padding_right\\":\\"6\\",\\"padding_opp_top\\":\\"false\\",\\"padding_top\\":\\"7\\",\\"b_sh_inset\\":\\"false\\",\\"b_sh_color\\":\\"#000000_0.10\\",\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_spread\\":\\"0\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_blur\\":\\"12\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_vOffset\\":\\"3\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"b_sh_hOffset\\":\\"0\\",\\"checkbox_b_ra_apply_all\\":\\"1\\",\\"b_ra_opp_left\\":\\"false\\",\\"b_ra_opp_bottom\\":\\"false\\",\\"b_ra_top\\":\\"8\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"18\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"link_color_hover\\":\\"#ff8fbc\\",\\"link_color\\":\\"#e24886\\",\\"margin_opp_left\\":\\"false\\",\\"margin_opp_top\\":\\"false\\",\\"custom_parallax_scroll_zindex\\":\\"3\\",\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"background_color\\":\\"#ffffff\\",\\"background_image-circle-radial\\":\\"false\\",\\"background_image-type\\":\\"image\\",\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"px\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_right\\":\\"50\\",\\"margin_top_unit\\":\\"px\\",\\"margin_bottom\\":\\"40\\",\\"breakpoint_mobile\\":{\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":\\"1\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"40\\",\\"margin_right_unit\\":\\"px\\",\\"margin_right\\":\\"30\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\",\\"margin_left\\":\\"30\\",\\"h3_margin_bottom_unit\\":\\"px\\",\\"h3_margin_top_unit\\":\\"px\\",\\"t_shh3_blur_unit\\":\\"px\\",\\"t_shh3_vShadow_unit\\":\\"px\\",\\"t_shh3_hShadow_unit\\":\\"px\\",\\"letter_spacing_h3_unit\\":\\"px\\",\\"line_height_h3_unit\\":\\"px\\",\\"font_size_h3_unit\\":\\"px\\",\\"font_size_h3\\":\\"16\\",\\"font_gradient_color_h3-circle-radial\\":\\"false\\",\\"font_gradient_color_h3-gradient-angle\\":\\"180\\",\\"font_gradient_color_h3-gradient-type\\":\\"linear\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\"},\\"breakpoint_tablet\\":{\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"px\\",\\"margin_left\\":\\"20\\",\\"margin_opp_left\\":\\"1\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"40\\",\\"margin_right_unit\\":\\"px\\",\\"margin_right\\":\\"20\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\"},\\"breakpoint_tablet_landscape\\":{\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"px\\",\\"margin_left\\":\\"20\\",\\"margin_opp_left\\":\\"1\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"40\\",\\"margin_right_unit\\":\\"px\\",\\"margin_right\\":\\"20\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\"},\\"background_image-gradient-angle\\":\\"180\\",\\"background_image-gradient-type\\":\\"linear\\"}}]}]}]',
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 345,
  'post_date' => '2020-01-27 07:47:35',
  'post_date_gmt' => '2020-01-27 07:47:35',
  'post_content' => '<!--themify_builder_static--><img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/teacher-training-a-500x280.jpg" width="500" height="280" alt="teacher-training-a" /><!--/themify_builder_static-->',
  'post_title' => 'Image Teacher Training',
  'post_excerpt' => '',
  'post_name' => 'tb_gs3455e2',
  'post_modified' => '2020-01-30 04:31:08',
  'post_modified_gmt' => '2020-01-30 04:31:08',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/tglobal-style/image-teacher-training/',
  'menu_order' => 0,
  'post_type' => 'tglobal_style',
  'meta_input' => 
  array (
    'themify_global_style_type' => 'image',
    'hide_page_title' => 'yes',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
    '_wp_old_slug' => 'image-teacher-training',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"row5e2e959740fe2\\",\\"cols\\":[{\\"element_id\\":\\"col5e2e959740fe2\\",\\"modules\\":[{\\"element_id\\":\\"mod5e2e959740fe2\\",\\"mod_name\\":\\"image\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"false\\",\\"param_image\\":\\"regular\\",\\"height_image\\":\\"280\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"500\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/teacher-training-a.jpg\\",\\"caption_on_overlay\\":\\"false\\",\\"style_image\\":\\"image-top\\",\\"margin_left\\":\\"-40\\",\\"margin_opp_left\\":\\"false\\",\\"margin_bottom\\":\\"-40\\",\\"margin_opp_top\\":\\"false\\",\\"padding_opp_left\\":\\"false\\",\\"padding_opp_top\\":\\"false\\",\\"custom_parallax_scroll_zindex\\":\\"1\\",\\"breakpoint_mobile\\":{\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"px\\",\\"margin_left\\":\\"0\\",\\"margin_opp_left\\":\\"false\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"-40\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\"},\\"breakpoint_tablet\\":{\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"px\\",\\"margin_left\\":\\"0\\",\\"margin_opp_left\\":\\"false\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"-40\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\"},\\"breakpoint_tablet_landscape\\":{\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"px\\",\\"margin_left\\":\\"0\\",\\"margin_opp_left\\":\\"false\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"-40\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\"}}}]}]}]',
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 328,
  'post_date' => '2020-01-27 04:35:31',
  'post_date_gmt' => '2020-01-27 04:35:31',
  'post_content' => '<!--themify_builder_static--><a href="https://themify.me/" > Sign Up </a><!--/themify_builder_static-->',
  'post_title' => 'Sign Up Class',
  'post_excerpt' => '',
  'post_name' => 'tb_gs3285e2',
  'post_modified' => '2020-02-28 00:55:19',
  'post_modified_gmt' => '2020-02-28 00:55:19',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/tglobal-style/sign-up-class/',
  'menu_order' => 0,
  'post_type' => 'tglobal_style',
  'meta_input' => 
  array (
    'themify_global_style_type' => 'buttons',
    'hide_page_title' => 'yes',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
    '_wp_old_slug' => 'sign-up-class',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"row5e2e6893d1752\\",\\"cols\\":[{\\"element_id\\":\\"col5e2e6893d1752\\",\\"modules\\":[{\\"element_id\\":\\"mod5e2e6893d1752\\",\\"mod_name\\":\\"buttons\\",\\"mod_settings\\":{\\"content_button\\":[{\\"label\\":\\"Sign Up\\",\\"link\\":\\"https:\\\\/\\\\/themify.me\\\\/\\",\\"link_options\\":\\"regular\\",\\"lightbox_width_unit\\":\\"px\\",\\"lightbox_height_unit\\":\\"px\\",\\"button_color_bg\\":\\"purple\\",\\"icon_alignment\\":\\"left\\"}],\\"display\\":\\"buttons-horizontal\\",\\"buttons_style\\":\\"solid\\",\\"buttons_shape\\":\\"squared\\",\\"buttons_size\\":\\"normal\\",\\"l_b_r_c_opp_left\\":\\"false\\",\\"l_b_r_c_opp_bottom\\":\\"false\\",\\"b_ra_opp_left\\":\\"false\\",\\"b_ra_opp_bottom\\":\\"false\\",\\"checkbox_link_margin_apply_all\\":\\"false\\",\\"link_margin_left_unit\\":\\"px\\",\\"link_margin_opp_left\\":\\"false\\",\\"link_margin_bottom_unit\\":\\"px\\",\\"link_margin_right_unit\\":\\"px\\",\\"link_margin_opp_top\\":\\"false\\",\\"link_margin_top_unit\\":\\"px\\",\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"%\\",\\"margin_opp_left\\":\\"false\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\",\\"text-shadow_blur_unit\\":\\"px\\",\\"text-shadow_vShadow_unit\\":\\"px\\",\\"text-shadow_hShadow_unit\\":\\"px\\",\\"font_style_regular\\":\\"normal\\",\\"letter_spacing_unit\\":\\"px\\",\\"line_height_unit\\":\\"px\\",\\"font_size_unit\\":\\"px\\",\\"font_gradient_color-circle-radial\\":\\"false\\",\\"font_gradient_color-gradient-angle\\":\\"180\\",\\"font_gradient_color-gradient-type\\":\\"linear\\",\\"font_color_type\\":\\"font_color_solid\\",\\"alignment\\":\\"center\\",\\"margin_left\\":\\"-24\\",\\"margin_top\\":\\"-18\\",\\"font_size\\":\\"14\\",\\"breakpoint_mobile\\":{\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"%\\",\\"margin_left\\":\\"0\\",\\"margin_opp_left\\":\\"false\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\",\\"margin_top\\":\\"-18\\"},\\"breakpoint_tablet\\":{\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"%\\",\\"margin_left\\":\\"-12\\",\\"margin_opp_left\\":\\"false\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\",\\"margin_top\\":\\"-18\\"}}}]}]}]',
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 226,
  'post_date' => '2020-01-13 08:42:36',
  'post_date_gmt' => '2020-01-13 08:42:36',
  'post_content' => '',
  'post_title' => 'Light Shadow',
  'post_excerpt' => '',
  'post_name' => 'tb_gs1305e1',
  'post_modified' => '2020-01-13 08:42:36',
  'post_modified_gmt' => '2020-01-13 08:42:36',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/tglobal-style/tb_gs1305e1/',
  'menu_order' => 0,
  'post_type' => 'tglobal_style',
  'meta_input' => 
  array (
    'themify_global_style_type' => 'image',
    'hide_page_title' => 'yes',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"row5e1c2d7ce96de\\",\\"styling\\":[],\\"cols\\":[{\\"element_id\\":\\"col5e1c2d7ce96de\\",\\"grid_class\\":\\"col-full\\",\\"styling\\":[],\\"modules\\":[{\\"element_id\\":\\"mod5e1c2d7ce96de\\",\\"mod_name\\":\\"image\\",\\"mod_settings\\":[{\\"element_id\\":\\"row5e1c26c687bcc\\",\\"styling\\":[],\\"cols\\":[{\\"element_id\\":\\"col5e1c26c687bcc\\",\\"grid_class\\":\\"col-full\\",\\"styling\\":[],\\"modules\\":[{\\"element_id\\":\\"mod5e1c26c687bcc\\",\\"mod_name\\":\\"image\\",\\"mod_settings\\":[{\\"element_id\\":\\"row5e1c25e0a598c\\",\\"styling\\":[],\\"cols\\":[{\\"element_id\\":\\"col5e1c25e0a598c\\",\\"grid_class\\":\\"col-full\\",\\"styling\\":[],\\"modules\\":[{\\"element_id\\":\\"mod5e1c25e0a598c\\",\\"mod_name\\":\\"image\\",\\"mod_settings\\":[{\\"element_id\\":\\"row5e17e42b7f464\\",\\"cols\\":[{\\"element_id\\":\\"col5e17e42b7f464\\",\\"modules\\":[{\\"element_id\\":\\"mod5e17e42b7f464\\",\\"mod_name\\":\\"image\\",\\"mod_settings\\":{\\"image_zoom_icon\\":\\"false\\",\\"param_image\\":\\"regular\\",\\"height_image\\":\\"309\\",\\"auto_fullwidth\\":\\"1\\",\\"width_image\\":\\"353\\",\\"appearance_image\\":\\"rounded\\",\\"url_image\\":\\"https:\\\\/\\\\/themify.me\\\\/demo\\\\/themes\\\\/ultra-yoga\\\\/files\\\\/2020\\\\/01\\\\/yoga-location-a.jpg\\",\\"caption_on_overlay\\":\\"false\\",\\"style_image\\":\\"image-top\\",\\"i_t_sh_inset\\":\\"false\\",\\"i_t_sh_spread_unit\\":\\"px\\",\\"i_t_sh_blur_unit\\":\\"px\\",\\"i_t_sh_vOffset_unit\\":\\"px\\",\\"i_t_sh_hOffset_unit\\":\\"px\\",\\"b_sh_inset\\":\\"false\\",\\"checkbox_i_t_r_c_apply_all\\":\\"1\\",\\"i_t_r_c_left_unit\\":\\"px\\",\\"i_t_r_c_opp_left\\":\\"false\\",\\"i_t_r_c_right_unit\\":\\"px\\",\\"i_t_r_c_bottom_unit\\":\\"px\\",\\"i_t_r_c_opp_bottom\\":\\"false\\",\\"i_t_r_c_top_unit\\":\\"px\\",\\"i_t_r_c_top\\":\\"6\\",\\"b_sh_color\\":\\"#000000_0.05\\",\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_spread\\":\\"5\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_blur\\":\\"5\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_vOffset\\":\\"4\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"b_sh_hOffset\\":\\"2\\"}}]}]}]}]}]}]}]}]}]}]}]}]',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 203,
  'post_date' => '2020-01-13 08:09:26',
  'post_date_gmt' => '2020-01-13 08:09:26',
  'post_content' => '<!--themify_builder_static--><img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-location-b-353x309.jpg" width="353" height="309" alt="yoga-location-b" /><!--/themify_builder_static-->',
  'post_title' => 'Image Shadow',
  'post_excerpt' => '',
  'post_name' => 'tb_gs2035e1',
  'post_modified' => '2020-01-30 03:36:35',
  'post_modified_gmt' => '2020-01-30 03:36:35',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/tglobal-style/image-shadow/',
  'menu_order' => 0,
  'post_type' => 'tglobal_style',
  'meta_input' => 
  array (
    'themify_global_style_type' => 'image',
    'hide_page_title' => 'yes',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
    '_wp_old_slug' => 'image-shadow',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"row5e1c25b617827\\",\\"cols\\":[{\\"element_id\\":\\"col5e1c25b617827\\",\\"modules\\":[{\\"element_id\\":\\"mod5e1c25b617827\\",\\"mod_name\\":\\"image\\",\\"mod_settings\\":{\\"content_text\\":\\"<h3>Down Town<\\\\/h3>\\\\n<p>604-682-3569<\\\\/p>\\\\n<p>888 Burrard Street, Vancouver, Australia, V6Z 1X9<\\\\/p>\\",\\"b_ra_opp_left\\":\\"false\\",\\"b_ra_opp_bottom\\":\\"false\\",\\"background_position\\":\\"50,50\\",\\"background_repeat\\":\\"repeat\\",\\"b_sh_inset\\":\\"false\\",\\"b_sh_color\\":\\"#000000_0.05\\",\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_spread\\":\\"5\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_blur\\":\\"5\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_vOffset\\":\\"4\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"b_sh_hOffset\\":\\"0\\",\\"checkbox_b_ra_apply_all\\":\\"false\\",\\"b_ra_left_unit\\":\\"px\\",\\"b_ra_left\\":\\"8\\",\\"b_ra_right_unit\\":\\"px\\",\\"b_ra_right\\":\\"8\\",\\"b_ra_bottom_unit\\":\\"px\\",\\"b_ra_bottom\\":\\"8\\",\\"b_ra_top_unit\\":\\"px\\",\\"b_ra_top\\":\\"8\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":\\"false\\",\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":\\"false\\",\\"padding_bottom_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":\\"false\\",\\"padding_top_unit\\":\\"px\\"}}}]}]}]',
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 41,
  'post_date' => '2020-01-09 11:42:47',
  'post_date_gmt' => '2020-01-09 11:42:47',
  'post_content' => '<!--themify_builder_static--><img src="https://themify.me/demo/themes/ultra-yoga/files/2020/01/yoga-pose-1.jpg" title="Floor Barre" alt="All Levels Welcome Beginner Friendly" /> <h3> Floor Barre </h3> All Levels Welcome<br/> Beginner Friendly<!--/themify_builder_static-->',
  'post_title' => 'Rounded Image Gallery',
  'post_excerpt' => '',
  'post_name' => 'tb_gs415e1',
  'post_modified' => '2020-02-10 22:36:59',
  'post_modified_gmt' => '2020-02-10 22:36:59',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/tglobal-style/rounded-image-gallery/',
  'menu_order' => 0,
  'post_type' => 'tglobal_style',
  'meta_input' => 
  array (
    'themify_global_style_type' => 'image',
    'hide_page_title' => 'yes',
    'themify_used_global_styles' => 
    array (
      0 => '',
    ),
    '_wp_old_slug' => 'rounded-image-gallery',
    '_themify_builder_settings_json' => '[{\\"element_id\\":\\"row5e1711b790327\\",\\"cols\\":[{\\"element_id\\":\\"col5e1711b790327\\",\\"modules\\":[{\\"element_id\\":\\"mod5e1711b790327\\",\\"mod_name\\":\\"image\\",\\"mod_settings\\":{\\"padding_bottom_unit\\":\\"%\\",\\"padding_top_unit\\":\\"%\\",\\"padding_opp_left\\":\\"false\\",\\"padding_bottom\\":\\"5\\",\\"padding_opp_top\\":\\"false\\",\\"padding_top\\":\\"0\\",\\"f_g_c_h3_h-circle-radial\\":\\"false\\",\\"f_c_t_h3_h\\":\\"f_c_h3_h_solid\\",\\"font_color_type_h3\\":\\"font_color_h3_solid\\",\\"hide_anchor\\":\\"false\\",\\"row_width\\":\\"fullwidth\\",\\"breakpoint_mobile\\":{\\"checkbox_padding_apply_all\\":\\"false\\",\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":\\"false\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"4\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":\\"false\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"0\\"},\\"checkbox_padding_apply_all\\":\\"false\\",\\"padding_left_unit\\":\\"px\\",\\"padding_right_unit\\":\\"px\\",\\"breakpoint_tablet\\":{\\"checkbox_padding_apply_all\\":\\"false\\",\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":\\"false\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"5\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":\\"false\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"0\\"},\\"breakpoint_tablet_landscape\\":{\\"checkbox_padding_apply_all\\":\\"false\\",\\"padding_left_unit\\":\\"px\\",\\"padding_opp_left\\":\\"false\\",\\"padding_bottom_unit\\":\\"%\\",\\"padding_bottom\\":\\"5\\",\\"padding_right_unit\\":\\"px\\",\\"padding_opp_top\\":\\"false\\",\\"padding_top_unit\\":\\"%\\",\\"padding_top\\":\\"0\\"},\\"checkbox_i_t_r_c_apply_all\\":\\"false\\",\\"i_t_r_c_left_unit\\":\\"px\\",\\"i_t_r_c_opp_left\\":\\"false\\",\\"i_t_r_c_right_unit\\":\\"px\\",\\"i_t_r_c_right\\":\\"8\\",\\"i_t_r_c_bottom_unit\\":\\"px\\",\\"i_t_r_c_opp_bottom\\":\\"false\\",\\"i_t_r_c_top_unit\\":\\"px\\",\\"i_t_r_c_top\\":\\"8\\",\\"i_t_b_left_style\\":\\"solid\\",\\"i_t_b_bottom_style\\":\\"solid\\",\\"i_t_b_right_style\\":\\"solid\\",\\"i_t_b_top_style\\":\\"solid\\",\\"i_t_b-type\\":\\"top\\",\\"checkbox_i_t_m_apply_all\\":\\"false\\",\\"i_t_m_left_unit\\":\\"px\\",\\"i_t_m_opp_left\\":\\"false\\",\\"i_t_m_bottom_unit\\":\\"px\\",\\"i_t_m_bottom\\":\\"20\\",\\"i_t_m_right_unit\\":\\"px\\",\\"i_t_m_opp_top\\":\\"false\\",\\"i_t_m_top_unit\\":\\"px\\",\\"t_sh_t_blur_unit\\":\\"px\\",\\"t_sh_t_vShadow_unit\\":\\"px\\",\\"t_sh_t_hShadow_unit\\":\\"px\\",\\"letter_spacing_title_unit\\":\\"px\\",\\"line_height_title_unit\\":\\"px\\",\\"font_size_title_unit\\":\\"px\\",\\"b_sh_inset\\":\\"false\\",\\"b_sh_color\\":\\"#000000_0.08\\",\\"b_sh_spread_unit\\":\\"px\\",\\"b_sh_blur_unit\\":\\"px\\",\\"b_sh_blur\\":\\"9\\",\\"b_sh_vOffset_unit\\":\\"px\\",\\"b_sh_vOffset\\":\\"3\\",\\"b_sh_hOffset_unit\\":\\"px\\",\\"b_sh_hOffset\\":\\"0\\",\\"checkbox_b_ra_apply_all\\":\\"1\\",\\"b_ra_left_unit\\":\\"px\\",\\"b_ra_opp_left\\":\\"false\\",\\"b_ra_right_unit\\":\\"px\\",\\"b_ra_bottom_unit\\":\\"px\\",\\"b_ra_opp_bottom\\":\\"false\\",\\"b_ra_top_unit\\":\\"px\\",\\"b_ra_top\\":\\"8\\",\\"checkbox_margin_apply_all\\":\\"false\\",\\"margin_left_unit\\":\\"px\\",\\"margin_opp_left\\":\\"false\\",\\"margin_bottom_unit\\":\\"px\\",\\"margin_bottom\\":\\"30\\",\\"margin_right_unit\\":\\"px\\",\\"margin_opp_top\\":\\"false\\",\\"margin_top_unit\\":\\"px\\",\\"link_color_hover\\":\\"#f57aac\\",\\"link_color\\":\\"#e24886\\"}}]}]}]',
  ),
  'tax_input' => 
  array (
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 150,
  'post_date' => '2020-01-10 06:19:29',
  'post_date_gmt' => '2020-01-10 06:19:29',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '150',
  'post_modified' => '2020-01-10 07:09:36',
  'post_modified_gmt' => '2020-01-10 07:09:36',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?p=150',
  'menu_order' => 1,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '5',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'main-navigation',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 174,
  'post_date' => '2020-01-10 07:09:30',
  'post_date_gmt' => '2020-01-10 07:09:30',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '174',
  'post_modified' => '2020-01-10 07:09:36',
  'post_modified_gmt' => '2020-01-10 07:09:36',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?p=174',
  'menu_order' => 2,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '153',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'main-navigation',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 173,
  'post_date' => '2020-01-10 07:09:30',
  'post_date_gmt' => '2020-01-10 07:09:30',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '173',
  'post_modified' => '2020-01-10 07:09:36',
  'post_modified_gmt' => '2020-01-10 07:09:36',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?p=173',
  'menu_order' => 3,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '159',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'main-navigation',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 172,
  'post_date' => '2020-01-10 07:09:30',
  'post_date_gmt' => '2020-01-10 07:09:30',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '172',
  'post_modified' => '2020-01-10 07:09:36',
  'post_modified_gmt' => '2020-01-10 07:09:36',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?p=172',
  'menu_order' => 4,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '161',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'main-navigation',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 171,
  'post_date' => '2020-01-10 07:09:30',
  'post_date_gmt' => '2020-01-10 07:09:30',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '171',
  'post_modified' => '2020-01-10 07:09:36',
  'post_modified_gmt' => '2020-01-10 07:09:36',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?p=171',
  'menu_order' => 5,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '163',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'main-navigation',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 170,
  'post_date' => '2020-01-10 07:09:30',
  'post_date_gmt' => '2020-01-10 07:09:30',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '170',
  'post_modified' => '2020-01-10 07:09:36',
  'post_modified_gmt' => '2020-01-10 07:09:36',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?p=170',
  'menu_order' => 6,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '165',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'main-navigation',
  ),
);
themify_process_post_import( $post );


$post = array (
  'ID' => 169,
  'post_date' => '2020-01-10 07:09:30',
  'post_date_gmt' => '2020-01-10 07:09:30',
  'post_content' => ' ',
  'post_title' => '',
  'post_excerpt' => '',
  'post_name' => '169',
  'post_modified' => '2020-01-10 07:09:36',
  'post_modified_gmt' => '2020-01-10 07:09:36',
  'post_content_filtered' => '',
  'post_parent' => 0,
  'guid' => 'https://themify.me/demo/themes/ultra-yoga/?p=169',
  'menu_order' => 7,
  'post_type' => 'nav_menu_item',
  'xfn' => '',
  'meta_input' => 
  array (
    '_menu_item_type' => 'post_type',
    '_menu_item_menu_item_parent' => '0',
    '_menu_item_object_id' => '167',
    '_menu_item_object' => 'page',
    '_menu_item_classes' => 
    array (
      0 => '',
    ),
  ),
  'tax_input' => 
  array (
    'nav_menu' => 'main-navigation',
  ),
);
themify_process_post_import( $post );



function themify_import_get_term_id_from_slug( $slug ) {
	$menu = get_term_by( "slug", $slug, "nav_menu" );
	return is_wp_error( $menu ) ? 0 : (int) $menu->term_id;
}

	$widgets = get_option( "widget_text" );
$widgets[1002] = array (
  'title' => '',
  'text' => 'Week Days: 07:00 – 17:00 · Saturday: 08:00 – 18:00 · Sunday: CLOSED',
  'filter' => true,
  'visual' => true,
);
update_option( "widget_text", $widgets );

$widgets = get_option( "widget_text" );
$widgets[1003] = array (
  'title' => '',
  'text' => 'Free Call: (+61) 123-716-6879',
  'filter' => true,
  'visual' => true,
);
update_option( "widget_text", $widgets );

$widgets = get_option( "widget_nav_menu" );
$widgets[1004] = array (
  'title' => 'Navigation',
  'nav_menu' => themify_import_get_term_id_from_slug( "main-navigation" ),
);
update_option( "widget_nav_menu", $widgets );

$widgets = get_option( "widget_text" );
$widgets[1005] = array (
  'title' => 'Opening Hours',
  'text' => 'Monday 07:00 -17:00

Tuesday 07:00-17:00

Wednesday 07:00-17:00

Thursday 07:00-17:00

Friday 07:00-17:00

Saturday 08:00-18:00

Sunday Closed',
  'filter' => true,
  'visual' => true,
);
update_option( "widget_text", $widgets );

$widgets = get_option( "widget_themify-social-links" );
$widgets[1006] = array (
  'title' => 'Social Media',
  'show_link_name' => NULL,
  'open_new_window' => NULL,
  'icon_size' => 'icon-medium',
  'orientation' => 'horizontal',
);
update_option( "widget_themify-social-links", $widgets );



$sidebars_widgets = array (
  'header-widget-1' => 
  array (
    0 => 'text-1002',
  ),
  'header-widget-2' => 
  array (
    0 => 'text-1003',
  ),
  'footer-widget-1' => 
  array (
    0 => 'nav_menu-1004',
  ),
  'footer-widget-2' => 
  array (
    0 => 'text-1005',
  ),
  'footer-widget-3' => 
  array (
    0 => 'themify-social-links-1006',
  ),
); 
update_option( "sidebars_widgets", $sidebars_widgets );

$homepage = get_posts( array( 'name' => 'home', 'post_type' => 'page' ) );
			if( is_array( $homepage ) && ! empty( $homepage ) ) {
				update_option( 'show_on_front', 'page' );
				update_option( 'page_on_front', $homepage[0]->ID );
			}
			$themify_data = array (
  'setting-webfonts_list' => 'recommended',
  'setting-default_layout' => 'sidebar1',
  'setting-default_post_layout' => 'list-post',
  'setting-post_filter' => 'no',
  'setting-disable_masonry' => 'yes',
  'setting-post_gutter' => 'gutter',
  'setting-default_layout_display' => 'content',
  'setting-default_more_text' => 'More',
  'setting-index_orderby' => 'date',
  'setting-index_order' => 'DESC',
  'setting-default_media_position' => 'above',
  'setting-image_post_feature_size' => 'blank',
  'setting-default_page_post_layout' => 'sidebar1',
  'setting-default_page_post_layout_type' => 'classic',
  'setting-default_page_single_media_position' => 'above',
  'setting-image_post_single_feature_size' => 'blank',
  'setting-search-result_layout_display' => 'content',
  'setting-search-result_media_position' => 'above',
  'setting-default_page_layout' => 'sidebar1',
  'setting-custom_post_tglobal_style_single' => 'sidebar1',
  'setting-default_portfolio_index_layout' => 'sidebar-none',
  'setting-default_portfolio_index_post_layout' => 'grid3',
  'setting-portfolio_post_filter' => 'yes',
  'setting-portfolio_disable_masonry' => 'yes',
  'setting-portfolio_gutter' => 'gutter',
  'setting-default_portfolio_index_display' => 'none',
  'setting-default_portfolio_index_title' => ' ',
  'setting-default_portfolio_index_unlink_post_title' => ' ',
  'setting-default_portfolio_index_hide_post_date' => ' ',
  'setting-default_portfolio_index_post_meta_category' => 'yes',
  'setting-default_portfolio_index_unlink_post_image' => 'yes',
  'setting-default_portfolio_single_layout' => 'sidebar-none',
  'setting-default_portfolio_single_portfolio_layout_type' => 'fullwidth',
  'setting-default_portfolio_single_title' => ' ',
  'setting-default_portfolio_single_unlink_post_title' => ' ',
  'setting-default_portfolio_single_hide_post_date' => ' ',
  'setting-default_portfolio_single_post_meta_category' => ' ',
  'setting-default_portfolio_single_unlink_post_image' => 'yes',
  'themify_portfolio_slug' => 'project',
  'themify_portfolio_category_slug' => 'portfolio-category',
  'setting-customizer_responsive_design_tablet_landscape' => '1280',
  'setting-customizer_responsive_design_tablet' => '768',
  'setting-customizer_responsive_design_mobile' => '680',
  'setting-mobile_menu_trigger_point' => '900',
  'setting-gallery_lightbox' => 'lightbox',
  'setting-header_design' => 'header-top-widgets',
  'setting-exclude_site_tagline' => 'on',
  'setting-exclude_search_form' => 'on',
  'setting_search_form' => 'live_search',
  'setting-exclude_rss' => 'on',
  'setting-header_widgets' => 'headerwidget-2col',
  'setting-footer_design' => 'footer-left-col',
  'setting-use_float_back' => 'on',
  'setting-footer_widgets' => 'footerwidget-3col',
  'setting-mega_menu_posts' => '5',
  'setting-mega_menu_image_width' => '180',
  'setting-mega_menu_image_height' => '120',
  'setting-imagefilter_applyto' => 'featuredonly',
  'setting-color_animation_speed' => '5',
  'setting-relationship_taxonomy' => 'category',
  'setting-relationship_taxonomy_entries' => '3',
  'setting-relationship_taxonomy_display_content' => 'none',
  'setting-single_slider_autoplay' => 'off',
  'setting-single_slider_speed' => 'normal',
  'setting-single_slider_effect' => 'slide',
  'setting-single_slider_height' => 'auto',
  'setting-more_posts' => 'infinite',
  'setting-entries_nav' => 'numbered',
  'setting-footer_text_left' => 'If you are looking for a balanced fitness regime in your life as well as a friendly environment and highly skilled instructors, then we are here to help. ',
  'setting-img_php_base_size' => 'large',
  'setting-global_feature_size' => 'blank',
  'setting-link_icon_type' => 'font-icon',
  'setting-link_type_themify-link-0' => 'image-icon',
  'setting-link_title_themify-link-0' => 'Twitter',
  'setting-link_img_themify-link-0' => 'https://themify.me/demo/themes/ultra-yoga/wp-content/themes/themify-ultra/themify/img/social/twitter.png',
  'setting-link_type_themify-link-1' => 'image-icon',
  'setting-link_title_themify-link-1' => 'Facebook',
  'setting-link_img_themify-link-1' => 'https://themify.me/demo/themes/ultra-yoga/wp-content/themes/themify-ultra/themify/img/social/facebook.png',
  'setting-link_type_themify-link-2' => 'image-icon',
  'setting-link_title_themify-link-2' => 'YouTube',
  'setting-link_img_themify-link-2' => 'https://themify.me/demo/themes/ultra-yoga/wp-content/themes/themify-ultra/themify/img/social/youtube.png',
  'setting-link_type_themify-link-3' => 'image-icon',
  'setting-link_title_themify-link-3' => 'Pinterest',
  'setting-link_img_themify-link-3' => 'https://themify.me/demo/themes/ultra-yoga/wp-content/themes/themify-ultra/themify/img/social/pinterest.png',
  'setting-link_type_themify-link-5' => 'font-icon',
  'setting-link_title_themify-link-5' => 'Facebook',
  'setting-link_link_themify-link-5' => 'https://www.facebook.com/themify',
  'setting-link_ficon_themify-link-5' => 'ti-facebook',
  'setting-link_type_themify-link-4' => 'font-icon',
  'setting-link_title_themify-link-4' => 'Twitter',
  'setting-link_link_themify-link-4' => 'https://www.twitter.com/themify',
  'setting-link_ficon_themify-link-4' => 'ti-twitter-alt',
  'setting-link_type_themify-link-6' => 'font-icon',
  'setting-link_title_themify-link-6' => 'YouTube',
  'setting-link_link_themify-link-6' => 'https://www.youtube.com/user/themifyme',
  'setting-link_ficon_themify-link-6' => 'ti-youtube',
  'setting-link_type_themify-link-7' => 'font-icon',
  'setting-link_title_themify-link-7' => 'Instagram',
  'setting-link_link_themify-link-7' => 'https://www.instagram.com/themify/?hl=en',
  'setting-link_ficon_themify-link-7' => 'ti-instagram',
  'setting-link_field_ids' => '{"themify-link-0":"themify-link-0","themify-link-1":"themify-link-1","themify-link-2":"themify-link-2","themify-link-3":"themify-link-3","themify-link-5":"themify-link-5","themify-link-4":"themify-link-4","themify-link-6":"themify-link-6","themify-link-7":"themify-link-7"}',
  'setting-link_field_hash' => '8',
  'setting-page_builder_is_active' => 'enable',
  'setting-page_builder_animation_appearance' => 'none',
  'setting-page_builder_animation_parallax_bg' => 'none',
  'setting-page_builder_animation_scroll_effect' => 'none',
  'setting-page_builder_animation_sticky_scroll' => 'none',
  'setting-page_builder_expiry' => '2',
  'skin' => 'yoga',
  'import_images' => 'on',
);
themify_set_data( $themify_data );
$theme = get_option( 'stylesheet' );
$theme_mods = array (
  0 => false,
  'custom_css_post_id' => -1,
);
update_option( "theme_mods_{$theme}", $theme_mods );
$menu_locations = array();
$menu = get_terms( "nav_menu", array( "slug" => "main-navigation" ) );
if( is_array( $menu ) && ! empty( $menu ) ) $menu_locations["main-nav"] = $menu[0]->term_id;
set_theme_mod( "nav_menu_locations", $menu_locations );



}
themify_do_demo_import();