<?php

use Illuminate\Foundation\Testing\WithoutMiddleware;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use ToStartWork\Qualification;

class QualificationUserTest extends TestCase
{
    use DatabaseMigrations;

    public function setUp()
    {
        parent::setUp();
        $this->withoutMiddleware();
    }

    /**
     * User is adding a new qualification to there account
     *
     */
    public function testUserStoringQualification()
    {
        $user = factory(\ToStartWork\User::class)->create();
        $qualification = factory(Qualification::class)->create();

        $data = [
            'ref_no' => '1234',
            'qualification_id' => $qualification->id,
            'issue_date' => '2017-01-01',
            'expiry_date' => '2018-01-01',
        ];

        $this->actingAs($user)
            ->json('POST', '/api/qualification/add', $data)
            ->seeJsonEquals([
                'qualification' => [
                    'id' => $qualification->id,
                    'is_approved' => 0,
                    'name' => $qualification->name,
                    'type_id' => $qualification->type_id,
                    'trade_type_id' => $qualification->trade_type_id,
                    'pivot' => [
                        'user_id' => $user->id,
                        'qualification_id' => $qualification->id,
                        'ref_no' => $data['ref_no'],
                        'issue_date' => $data['issue_date'],
                        'expiry_date' => $data['expiry_date']
                    ]
                ],
            ]);
    }

    public function testUserRemovingQualification()
    {
        $user = factory(\ToStartWork\User::class)->create();
        $qualification = factory(Qualification::class)->create();
        $user->qualifications()->attach($qualification->id, ['ref_no' => '1234', 'issue_date' => '2017-01-01', 'expiry_date' => '2018-01-01']);
        $this->assertEquals(1, $user->qualifications()->count());


        $this->actingAs($user)
            ->json('POST', sprintf('/api/qualification/%d/remove', $qualification->id), [])
            ->seeStatusCode(200);

        $this->assertEquals(0, $user->qualifications()->count());
    }

    public function testUserUpdatingQualification()
    {
        $user = factory(\ToStartWork\User::class)->create();
        $qualification = factory(Qualification::class)->create();
        $user->qualifications()->attach($qualification->id, ['ref_no' => '1234', 'issue_date' => '2017-01-01', 'expiry_date' => '2018-01-01']);

        $data = [
            'ref_no' => '12345',
            'qualification_id' => $qualification->id,
            'issue_date' => '2018-01-01',
            'expiry_date' => '2019-01-01',
        ];

        $this->actingAs($user)
            ->json('POST', sprintf('/api/qualification/%d/update', $qualification->id), $data)
            ->seeJsonEquals([
                'qualification' => [
                    'id' => $qualification->id,
                    'is_approved' => 0,
                    'name' => $qualification->name,
                    'type_id' => $qualification->type_id,
                    'trade_type_id' => $qualification->trade_type_id,
                    'pivot' => [
                        'user_id' => $user->id,
                        'qualification_id' => $qualification->id,
                        'ref_no' => $data['ref_no'],
                        'issue_date' => $data['issue_date'],
                        'expiry_date' => $data['expiry_date']
                    ]
                ],
            ]);
    }

    public function testUserQualificationsList()
    {
        $user = factory(\ToStartWork\User::class)->create();
        $qualification = factory(Qualification::class)->create();
        $user->qualifications()->attach($qualification->id, ['ref_no' => '1234', 'issue_date' => '2017-01-01', 'expiry_date' => '2018-01-01']);

        $this->actingAs($user)
            ->json('GET', '/api/qualification/list')
            ->seeJsonEquals([
                'qualifications' => [[
                    'id' => $qualification->id,
                    'name' => $qualification->name,
                    'type_id' => $qualification->type_id,
                    'trade_type_id' => $qualification->trade_type_id,
                    'ref_no' => '1234',
                    'issue_date' => '2017-01-01',
                    'expiry_date' => '2018-01-01',
                    'front_image_url' => null,
                    'back_image_url' => null
                ]],
            ]);
    }
}
